/** 
 * @file    kForwardBlockInfo.h
 * @brief   Declares the kForwardBlockInfo class. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_FORWARD_BLOCK_INFO_H
#define K_FIRESYNC_FORWARD_BLOCK_INFO_H

#include <kFireSync/kNodeDef.h>

#define kFORWARD_BLOCK_INFO_MAX_PORT_COUNT         (64)      ///< Maximum supported channels.

/**
 * @class   kForwardBlockInfo
 * @extends kObject
 * @ingroup kFireSync-Blocks
 * @brief   Represents configuration for kForwardBlock. 
 */
//typedef kObject kForwardBlockInfo;        --forward-declared in kFsDef.x.h

/** 
 * Constructs a kForwardBlockInfo object. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Receives constructed info object. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
 kFsFx(kStatus) kForwardBlockInfo_Construct(kForwardBlockInfo* info, kAlloc allocator); 

/** 
 * Constructs a kForwardBlockInfo object and loads its content from an XML object. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Receives constructed info object.  
 * @param   xml         Source from which parameters are loaded. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_Load(kForwardBlockInfo* info, kXml xml, kAlloc allocator); 

/** 
 * Constructs a kXml object and uses it to store the content of this kForwardBlockInfo instance.
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @param   xml         Receives constructed kXml object.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_Store(kForwardBlockInfo info, kXml* xml, kAlloc allocator); 

/** 
 * Reads kForwardBlockInfo state from a kXml object. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_Read(kForwardBlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Writes kForwardBlockInfo state to a kXml object. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_Write(kForwardBlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Checks and automatically corrects all state variables. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_Validate(kForwardBlockInfo info); 

/** 
 * Resets all state variables to default values.
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_Clear(kForwardBlockInfo info); 

/** 
 * Sets the count of input/output port pairs. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @param   portCount   Count of ports to be created.
 * @return              Operation status. 
 */
kFsFx(kStatus) kForwardBlockInfo_SetPortCount(kForwardBlockInfo info, kSize portCount); 

/** 
 * Gets the count of input/output port pairs. 
 *
 * @public              @memberof kForwardBlockInfo
 * @param   info        Info object.  
 * @return              Count of ports to be created.
 */
kFsFx(kSize) kForwardBlockInfo_PortCount(kForwardBlockInfo info); 

/** Gets the input port id associated with the specified input/output pair index. */
kInlineFx(k16u) kForwardBlockInfo_InputId(kSize index)  { return (k16u) index; }

/** Gets the output port id associated with the specified input/output pair index. */
kInlineFx(k16u) kForwardBlockInfo_OutputId(kSize index) { return (k16u) (kFORWARD_BLOCK_INFO_MAX_PORT_COUNT + index); }

/** Gets the input/output pair index associated with the specified input port id. */
kInlineFx(kSize) kForwardBlockInfo_InputIndex(k32u id) { return id; }

/** Gets the input/output pair index associated with the specified output port id. */
kInlineFx(kSize) kForwardBlockInfo_OutputIndex(k32u id) { return id - kFORWARD_BLOCK_INFO_MAX_PORT_COUNT; }

#include <kFireSync/Blocks/kForwardBlockInfo.x.h>

#endif
