/** 
 * @file    kBinCSum4BlockInfo.h
 * @brief   Declares the kBinCSum4BlockInfo class. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_BIN_CSUM4_BLOCK_INFO_H
#define K_FIRESYNC_BIN_CSUM4_BLOCK_INFO_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kBinCSum4BlockInfo
 * @extends kObject
 * @ingroup kFireSync-Blocks
 * @brief   Represents configuration for kBinCSum4Block. 
 */
//typedef kObject kBinCSum4BlockInfo;        --forward-declared in kFsDef.x.h

/** 
 * Constructs a kBinCSum4BlockInfo object. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Receives constructed info object. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
 kFsFx(kStatus) kBinCSum4BlockInfo_Construct(kBinCSum4BlockInfo* info, kAlloc allocator); 

/** 
 * Constructs a kBinCSum4BlockInfo object and loads its content from an XML object. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Receives constructed info object.  
 * @param   xml         Source from which parameters are loaded. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_Load(kBinCSum4BlockInfo* info, kXml xml, kAlloc allocator); 

/** 
 * Constructs a kXml object and uses it to store the content of this kBinCSum4BlockInfo instance.
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   xml         Receives constructed kXml object.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_Store(kBinCSum4BlockInfo info, kXml* xml, kAlloc allocator); 

/** 
 * Reads kBinCSum4BlockInfo state from a kXml object. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_Read(kBinCSum4BlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Writes kBinCSum4BlockInfo state to a kXml object. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_Write(kBinCSum4BlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Checks and automatically corrects all state variables. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_Validate(kBinCSum4BlockInfo info); 

/** 
 * Resets all state variables to default values.
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_Clear(kBinCSum4BlockInfo info); 

/** 
 * Sets the binarize threshold. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   threshold   Binarize threshold. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_SetBinThreshold(kBinCSum4BlockInfo info, k32u threshold); 

/** 
 * Reports the binarize threshold. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object. 
 * @return              Binarize threshold. 
 */
kFsFx(k32u) kBinCSum4BlockInfo_BinThreshold(kBinCSum4BlockInfo info);

/** 
 * Sets the column-sum threshold. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   threshold   Column-sum threshold. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_SetCSumThreshold(kBinCSum4BlockInfo info, k32u threshold); 

/** 
 * Reports the column-sum threshold. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object. 
 * @return              Column-sum threshold. 
 */
kFsFx(k32u) kBinCSum4BlockInfo_CSumThreshold(kBinCSum4BlockInfo info);

/** 
 * Sets the number of left-most columns used to calculate background noise.  
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   width       Column count, prior to any subsampling.
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_SetBackgroundWidth(kBinCSum4BlockInfo info, kSize width); 

/** 
 * Reports the number of left-most columns used to calculate background noise.  
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object. 
 * @return              Column count, prior to any subsampling.
 */
kFsFx(kSize) kBinCSum4BlockInfo_BackgroundWidth(kBinCSum4BlockInfo info);

/** 
 * Sets the number of active camera algorithm "pipes" that should be simulated. 
 * 
 * "Pipes" are a PL implementation detail. Unfortunately, the number of pipes can subtley affect 
 * algorithm results. If you are attempting to <em>precisely</em> match the PL algorithm, you
 * will need to consult the PL camera module documentation for details.  If approximate results 
 * will suffice, then set this value to 1. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object.  
 * @param   count       Active pipe count.
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4BlockInfo_SetActivePipeCount(kBinCSum4BlockInfo info, kSize count); 

/** 
 * Reports the number of active camera algorithm "pipes" that should be simulated. 
 *
 * @public              @memberof kBinCSum4BlockInfo
 * @param   info        Info object. 
 * @return              Active pipe count.
 */
kFsFx(kSize) kBinCSum4BlockInfo_ActivePipeCount(kBinCSum4BlockInfo info);

/** Gets the input port id associated with the specified input/output pair index. */
kInlineFx(k16u) kBinCSum4BlockInfo_InputId()    { return (k16u)0; }

/** Gets the output port id associated with the specified input/output pair index. */
kInlineFx(k16u) kBinCSum4BlockInfo_OutputId()   { return (k16u)1; }

#include <kFireSync/Blocks/kBinCSum4BlockInfo.x.h>

#endif
