/** 
 * @file    kInvertAlg.h
 * @brief   Declares the kInvertAlg type. 
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_INVERT_ALG_H
#define K_FIRESYNC_INVERT_ALG_H

#include <kFireSync/kFsDef.h>

/**
 * @struct  kInvertAlgParams
 * @extends kValue
 * @ingroup kFireSync-Blocks-Details
 * @brief   Input parameters for kInvertAlgorithm.
 */
typedef struct kInvertAlgParams
{
    kType algorithmType;        ///< Algorithm implementation to be used.
    kAlloc objectAlloc;         ///< Object memory allocator for outputs.
    kAlloc valueAlloc;           ///< Data memory allocator for outputs.
} kInvertAlgParams; 

#include <kFireSync/Blocks/Details/kInvertAlg.x.h>

/**
 * @class   kInvertAlg
 * @extends kObject
 * @ingroup kFireSync-Blocks-Details
 * @brief   Abstract base class for image inversion algorithm implementations. 
 */
//typedef kObject kInvertAlg;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a specific kInvertAlg child instance.
 *
 * @public              @memberof kInvertAlg
 * @param   algorithm   Destination for the constructed object handle. 
 * @param   params      Algorithm parameters. 
 * @param   allocator   Memory allocator for object instance (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kInvertAlg_Construct(kInvertAlg* algorithm, const kInvertAlgParams* params, kAlloc allocator); 

/** 
 * Executes the invert operation. 
 *
 * @public                  @memberof kInvertAlg
 * @param   algorithm       kInvertAlg object.  
 * @param   input           Image to be inverted.
 * @param   inputExclusive  Is the input array exclusively owned (known to be not shared)?
 * @param   output          Receives inverted output image.
 * @return                  Operation status. 
 */
kInlineFx(kStatus) kInvertAlg_Execute(kInvertAlg algorithm, kImage input, kBool inputExclusive, kImage* output)
{
    return xkInvertAlg_VTable(algorithm)->VExecute(algorithm, input, inputExclusive, output);
}

#endif
