/** 
 * @file    GvUtils.h
 * @brief   GoVision utilities. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef GV_UTILS_H
#define GV_UTILS_H

#include <GoVision/GvDef.h>
#include <kApi/Data/kXml.h>
#include <kFireSync/Data/kMsg.h>
#include <kFireSync/Pipe/kPxPort.h>
#include <kFireSync/Pipe/kPxBlock.h>
#include <kApi/Data/kArray2.h>

#define GV_UTILS_EPS_X      (0.0005)    // 0.5 micron
#define GV_UTILS_EPS_Y      (0.0005)    // 0.5 micron
#define GV_UTILS_EPS_Z      (0.00005)   // 0.05 micron
#define GV_UTILS_EPS_ANGLE  (0.001)     // 0.001 degrees
#define GV_UTILS_EPS_SIN_ANGLE  (0.0000175)     // sin(0.001 degrees)

#define GV_SURFACEMSG_POSE_ID_SENSOR        (0)
#define GV_SURFACEMSG_POSE_ID_SURFACE        (1)
#define GV_SURFACEMSG_POSE_ID_SOURCEPART    (2)

#define GV_PROFILEMSG_POSE_ID_SOURCE        (1000)

#define GV_UTILS_FLOAT_EQUAL(v0, v1, eps) ((v0) > (v1) ? (v0) - (v1) < eps : (v1) - (v0) < eps)

// DEPRECATED ---------------------------------------------------------------------------------//
// GOC-14326 Deprecated due to its bad performance when V is large.
// Please use the newer GV_UTILS_POSITIVE_MOD1 instead.
// Kept for unit test purpose only.
// Note the function name is swapped with the newer implementation so new change is required for any existing callers.
static __inline kSize GV_UTILS_POSITIVE_MOD1(k64s V, kSize M)
{
    k32s sM = (k32s)M;
    k64u uV;

    while (V < 0) V += sM;
    uV = (k64u)V;
    return (kSize)(uV % M);
}

// GOC-14326 Find the smallest positive modulo.
// This is not equivalent to % operator.
// V % M is typically implemented as a remainder operator.
// Example : (-10) % (7) = -3, GV_UTILS_POSITIVE_MOD1(-10, 7) = 4;
// Note that V is a signed 64bit integer. M is a unsigned integer (kSize). 
// If M is any non-natural number, its return value is undefined behavior.
static __inline kSize GV_UTILS_POSITIVE_MOD(k64s V, kSize M)
{
    // Must cast to k64s to calculate correctly.
    k64s sM = (k64s)M;
    return (kSize)((V % sM + sM) % sM);
}

static __inline k32s GV_UTILS_FLOOR_32S(k64f x, k64f eps)
{
   k32s   ix = (k32s)x;

   if (x > -0.0)
   {
      if (x + eps > ix + 1)
          return ix + 1;
      else
          return ix;
   }
   else
   {
      if (x + eps > ix)
          return ix;
      else
          return ix - 1;
   }
}

static __inline k32s GV_UTILS_CEIL_32S(k64f x, k64f eps)
{
   k32s   ix = (k32s)x;

   if (x <= -0.0)
   {
       if (x - eps < ix - 1)
           return ix - 1;
       else
           return ix;
   }
   else
   {
       if (x - eps < ix)
           return ix;
       else
           return ix + 1;
   }
}

/* Pose Stuff */
typedef struct kRect3d32s 
{
    k32s x;
    k32s y;
    k32s z;
    k32s width;
    k32s height;
    k32s depth;
} kRect3d32s;

typedef struct kPose2d64f 
{
    k64f angle;
    k64f x;
    k64f y;
} kPose2d64f;

GvFx(kStatus) GvUtils_PoseInverse(kPose2d64f* inversePose, kPose2d64f* srcPose);
GvFx(kStatus) GvUtils_PoseRelative(kPose2d64f* relativePose, kPose2d64f* fromPose, kPose2d64f* toPose);
GvFx(kStatus) GvUtils_UpdatePose(kPose2d64f* updatePose, const kPose2d64f* pose);

GvFx(kStatus) GvUtils_LocalToGlobal2d(kPose2d64f* pose, k64f x, k64f y, k64f* xGlobal, k64f* yGlobal);
GvFx(kStatus) GvUtils_GlobalToLocal2d(kPose2d64f* pose, k64f x, k64f y, k64f* xLocal, k64f* yLocal);

/* Feature Stuff */
typedef struct GvProtoFeature 
{
    GvFeatureType type;
    kPoint3d64f position;
    kPoint3d64f orientation;
} GvProtoFeature;

typedef GvProtoFeature GvProtoFeaturePoint;
typedef GvProtoFeature GvProtoFeaturePlane;
typedef GvProtoFeature GvProtoFeatureLine;

typedef struct GvProtoFeatureCircle
{
    GvFeatureType type;
    kPoint3d64f position;
    kPoint3d64f orientation;
    k64f radius;
} GvProtoFeatureCircle;

//Set an array to appropriate k*_NULL value
GvFx(kStatus) GvUtils_Array1Null(kArray1 a1);

GvFx(kSize) GvUtils_ProtoFeatureSize(GvFeatureType featureType);
GvFx(GvFeatureType) GvUtils_ProtoFeatureType(const void* feature);

GvFx(kStatus) GvUtils_NearestPointOnLine2d(kPoint64f* point, k64f angle, k64f distance, kPoint64f* nearestPoint);
GvFx(kStatus) GvUtils_NearestPointOnPlaneTilt(kPoint3d64f* point, const kPoint64f* tilt, k64f distance, kPoint3d64f* nearestPoint);

/* Anchor Stuff */

typedef struct GvSurfaceAnchorInfo
{
    kBool valid;
    kPoint3d64f X;
    kPoint3d64f Y;
    kPoint3d64f Z;
    GvAngleAnchor line;
    kPoint3d64f referencePoint;
} GvSurfaceAnchorInfo;

GvFx(kStatus) GvUtils_PrepareSurfaceCommonOutput(k64f value, GvAngleAnchor* anchorOut, GvSurfaceAnchorInfo* anchorInfo, k64f rotationAngle, GvSurfaceCommonOutput* output);
GvFx(kStatus) GvUtils_UpdateSurfaceAnchorInfo(kMsg anchorXMsg, kMsg anchorYMsg, kMsg anchorZMsg, kMsg anchorZAngleMsg, GvSurfaceAnchorInfo* anchorInfo);

GvFx(kStatus) GvUtils_RectToRegion(const kRect3d64f* rectIn, const kPoint3d64f* orientationIn, GvRegion3d64f* regionOut);
GvFx(kStatus) GvUtils_RegionToRect(const GvRegion3d64f* regionIn, kRect3d64f* rectOut, kPoint3d64f* orientationOut);
GvFx(kStatus) GvUtils_RegionToXYPoints(const GvRegion3d64f* regionIn, kPoint64f* pointsOut);

GvFx(kStatus) GvUtils_AnchorRegion(GvSurfaceAnchorInfo* anchorInfo, k64f referenceAngle, GvAnchorCoRType coRType, kBool snapToAngle, 
    GvRegion3d64f* regionIn, GvRegion3d64f* regionOut);
GvFx(kStatus) GvUtils_SurfacePrepareOutput(k64f value, kBool anchorValid, kPoint3d64f* anchorSource, kPoint3d64f* anchorPosition, k64f anchorAngle, GvSurfaceMeasurementBase* output);
GvFx(kBool) GvUtils_SurfaceAnchorInfo(kMsg anchorXMsg, kMsg anchorYMsg, kMsg anchorZMsg, kMsg anchorZAngleMsg,
    kPoint3d64f* anchor, kPoint3d64f* anchorPosition, k64f* anchorZAngle);

GvFx(kStatus) GvUtils_ProfilePrepareOutput(k64f value, kBool anchorValid, kPoint64f* anchorSource, kPoint64f* anchorPosition, GvProfileMeasurementBase* output);

GvFx(kBool) GvUtils_ProfileAnchorInfo(kMsg anchorXMsg, kMsg anchorZMsg, kPoint64f* anchor, kPoint64f* anchorPosition);

GvFx(kStatus) GvUtils_ParseProfileMsg(kMsg msg, kArray1* ranges, k64f sensorOffset, k64f* offset);
GvFx(kStatus) GvUtils_ParseSurfaceMsg(kObject msg, GvSurfaceFrameOfReferenceType refType, const kPoint64f* sensorOffsets, kArray2* ranges, kPoint64f* offsets, kPoint3d64f* resolution);
GvFx(kStatus) GvUtils_ParseSectionMsg(kMsgSet msg, kSize index, kBool getIntensity, kArray1* dataOut, const kPoint3d64f** offset, const kPoint3d64f** scale, const kPose2d64f** pose);

GvFx(kStatus) GvUtils_Swap(void* left, void* right, kType type);

GvFx(kStatus) GvUtils_WritePoint3d64f(const kPoint3d64f* point, kXml xml, kXmlItem item);
GvFx(kStatus) GvUtils_ReadPoint3d64f(kPoint3d64f* point, kXml xml, kXmlItem item);

// GvUtils_Roi32s is obsolete and should be replaced by GvUtils_Roi2d32s
GvFx(kStatus) GvUtils_Roi32s(const kRect64f *fRoi, k64f fovX, k64f xResolution, k64f zResolution, kRect32s *iRoi);
GvFx(kStatus) GvUtils_Roi2d32s(const kRect64f* fRoi, const kPoint64f* dataOrigin, k64f xResolution, k64f zResolution, kRect32s* iRoi);
GvFx(kStatus) GvUtils_Roi3d32s(const kRect3d64f* fRoi, const kPoint3d64f* dataOrigin, 
                               k64f xResolution, k64f yResolution, k64f zResolution, kRect3d32s* iRoi);
GvFx(kStatus) GvUtils_WriteRect3d64f(const kRect3d64f* rect, kXml xml, kXmlItem item);
GvFx(kStatus) GvUtils_ReadRect3d64f(kRect3d64f* rect, kXml xml, kXmlItem item);

GvFx(kStatus) GvUtils_BoundRegion2d(const kRect64f* inRegion, const kRect64f* fov, kRect64f* outRegion);
GvFx(kStatus) GvUtils_BoundOffsetRegion2d(const kRect64f* inRegion, const kPoint64f* offset, const kRect64f* fov, kRect64f* outRegion);

GvFx(kStatus) GvUtils_DataRegion(kObject surface, const kPoint3d64f* dataOrigin, const kPoint3d64f* scale, kRect3d64f* dataRegion);
GvFx(kStatus) GvUtils_OffsetDataRegion3d(const kRect3d64f* inRegion,
    kArray2 surface, const kPoint3d64f* offset, const kPoint3d64f* dataOrigin, const kPoint3d64f* scale,
    kRect3d64f* outRegion);
GvFx(kStatus) GvUtils_BoundOffsetRegion3d(const kRect3d64f* inRegion, const kPoint3d64f* offset, const kRect3d64f* dataRegion, kRect3d64f* outRegion);

GvFx(kStatus) GvUtils_SliceMsgSet(kMsgSet input, kMsgSet* output, kSize start, kSize count, kAlloc allocator);
GvFx(kStatus) GvUtils_SliceMsgSetEx(kMsgSet input, kMsgSet* output, kArrayList selectedIndices, kAlloc allocator);

GvFx(kStatus) GvUtils_ConstructMsgSet(kMsgSet* output, kMsgSet prototype, kSize count, kAlloc allocator);
GvFx(kStatus) GvUtils_CopyMsgSet(kMsgSet dst, kSize dstStart, kMsgSet src, kSize srcStart, kSize count);

GvFx(k64u) GvUtils_ConvertSIMicrosecondsToFSMicroseconds(k64u input_SI);
GvFx(k64u) GvUtils_ConvertFSMicroSecondstoMicroseconds(k64u input_FS);

GvFx(k32u) GvUtils_CalcNumResampledProfilePointsInRange(k64f range, k64f step);


// Deprecated kPxBlock functions
/// @cond IGNORE_DOXYGEN_CONFUSED
kInlineFx(kStatus) GvUtils_PxBlockFree(kPxBlock block, kMsgSet msg)
{
    return kObject_Dispose(msg);
}
kInlineFx(kPxPort) GvUtils_PxBlockPort(kPxBlock block, k32u id)
{
    kPxPort port = kNULL; 
    return kSuccess(kPxBlock_FindPort(block, id, &port)) ? port : kNULL;
}
kInlineFx(kBool) GvUtils_PxBlockIsRouted(kPxBlock block, kPxPort port)
{
    return kPxPort_IsRouted(port);
}
kInlineFx(void*) GvUtils_PxBlockContext(kPxBlock block)
{
    return block;
}
/// @endcond

#endif
