/**
 * @file    GvDef.h
 * @brief   Essential GoVision declarations.
 *
 * @internal
 * Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef GV_DEF_H
#define GV_DEF_H

#include <kApi/kApiDef.h>

#if defined(GV_EMIT)
#    define GvFx(TYPE)       kExportFx(TYPE)
#    define GvDx(TYPE)       kExportDx(TYPE)
#else
#    define GvFx(TYPE)       kImportFx(TYPE)
#    define GvDx(TYPE)       kImportDx(TYPE)
#endif

// TODO: remove these definitions once VE repo defines their own version of these.
#ifndef pgettext
#define pgettext(C, T)          T
#endif
#ifndef _
#define _(T)                    T
#endif

#define GV_FIRST_FRAME_NUMBER                   (1)
#define GV_ANGLE_LUT_PRECISION                  (12)

#define SENSOR_LINE_COUNT                        (1)
#define SENSOR_LUT_CENTRE_STEP                   (256)
#define SENSOR_ALIGN_CAL_SAMPLE_COUNT            (10)
#define SENSOR_ALIGN_CAL_MIN_SLOT_WIDTH          (2.5)

#define GV_MAX_EXPOSURE_STEPS                    (5)

typedef k32s GvSpotSelectionType;

#define GV_SPOT_SELECTION_INTENSITY             (0)
#define GV_SPOT_SELECTION_TOP                   (1)
#define GV_SPOT_SELECTION_BOTTOM                (2)
#define GV_SPOT_SELECTION_NONE                  (3)
#define GV_SPOT_SELECTION_CONTINUITY_SORT       (4)
#define GV_SPOT_SELECTION_TRANSLUCENT_SORT      (5)

typedef k32s GvTriggerSource;

#define GV_TRIGGER_SOURCE_TIME             (0)
#define GV_TRIGGER_SOURCE_ENCODER          (1)
#define GV_TRIGGER_SOURCE_OTHER            (2)

typedef k32s GvMovementDirection;

#define GV_MOVEMENT_UNKNOWN                 (0)
#define GV_MOVEMENT_FORWARD                 (1)
#define GV_MOVEMENT_BACKWARD                (2)

#define GV_MINIMUM_ANGLE_STEP                   (0.001) // millidegrees

#define GV_TOOL_INPUT_TOP_NAME              "InTop"
#define GV_TOOL_INPUT_BOTTOM_NAME           "InBottom"
#define GV_TOOL_INPUT_OFFSET_X_NAME         "InOffsetX"
#define GV_TOOL_INPUT_OFFSET_Y_NAME         "InOffsetY"
#define GV_TOOL_INPUT_OFFSET_Z_NAME         "InOffsetZ"
#define GV_TOOL_INPUT_OFFSET_ZANGLE_NAME    "InOffsetZAngle"
#define GV_TOOL_INPUT_FOR_NAME              "InFoR"
#define GV_TOOL_INPUT_FEATURE_NAME          "InFeature"
#define GV_TOOL_INPUT_MAX_FEATURES          (2)

typedef enum
{
    GV_TOOL_INPUT_TOP           = 0,
    GV_TOOL_INPUT_BOTTOM        = 1,
    GV_TOOL_INPUT_OFFSET_X      = 2,
    GV_TOOL_INPUT_OFFSET_Y      = 3,
    GV_TOOL_INPUT_OFFSET_Z      = 4,
    GV_TOOL_INPUT_OFFSET_ZANGLE = 5,
    GV_TOOL_INPUT_FOR           = 6,
    GV_TOOL_INPUT_FEATURE_START = 100
} GvToolInputBlockPort;

typedef struct
{
    k64f value;
    k64f unfilteredValue;
    k64f decision;
} GvMeasurementBase;

typedef struct
{
    GvMeasurementBase base;
    kPoint64f anchorSource;
    kPoint64f anchorPosition[2];
} GvProfileMeasurementBase;

typedef struct
{
    GvMeasurementBase base;
    kPoint3d64f anchorSource;
    kPoint3d64f anchorPosition[4];
    k64f anchorAngle;
} GvSurfaceMeasurementBase;

typedef k32s GvFeatureType;
#define GV_FEATURE_POINT    (0)
#define GV_FEATURE_LINE     (1)
#define GV_FEATURE_CIRCLE   (2)
#define GV_FEATURE_PLANE    (3)


typedef k32s GvAnchorCoRType;
#define GV_ANCHOR_COR_RIGID_BODY    (0)
#define GV_ANCHOR_COR_AXIS_ALIGNED  (1)
#define GV_ANCHOR_COR_REGION_CENTER (2)

typedef struct GvAngleAnchor
{
    kPoint3d64f position;
    k64f angle;
} GvAngleAnchor;

typedef struct GvSurfaceCommonOutput
{
    k64f value;
    k64f decision;
    GvAngleAnchor anchorOut;
    GvAngleAnchor anchorIn;
} GvSurfaceCommonOutput;

typedef struct GvRegion3d64f
{
    kPoint3d64f position;
    kPoint3d64f dimension;
    kPoint3d64f orientation;
} GvRegion3d64f;

kDeclareValueEx(Gv, GvRegion3d64f, kValue)

#define GV_INIT_NULL_REGION   { { k64F_NULL, k64F_NULL, k64F_NULL }, { k64F_NULL, k64F_NULL, k64F_NULL }, { k64F_NULL, k64F_NULL, k64F_NULL } }

typedef enum
{
    // Position results  types
    GV_SURFACE_FEATURE_TYPE_AVERAGE = 0,
    GV_SURFACE_FEATURE_TYPE_MEDIAN,
    GV_SURFACE_FEATURE_TYPE_CENTROID,
    GV_SURFACE_FEATURE_TYPE_MAX_X,
    GV_SURFACE_FEATURE_TYPE_MIN_X,
    GV_SURFACE_FEATURE_TYPE_MAX_Y,
    GV_SURFACE_FEATURE_TYPE_MIN_Y,
    GV_SURFACE_FEATURE_TYPE_MAX_Z,
    GV_SURFACE_FEATURE_TYPE_MIN_Z,
    // Height result types
    GV_SURFACE_FEATURE_TYPE_MEAN_HEIGHT,
    GV_SURFACE_FEATURE_TYPE_MEDIAN_HEIGHT,
    GV_SURFACE_FEATURE_TYPE_MAX_HEIGHT,
    GV_SURFACE_FEATURE_TYPE_MIN_HEIGHT,
    GV_SURFACE_FEATURE_TYPE_CENTROID_2D_HEIGHT,
    GV_SURFACE_FEATURE_TYPE_CENTROID_3D_HEIGHT
} GvSurfaceFeatureType;

typedef enum
{
    SENSOR_ALIGN_DIRECTION_FORWARD,
    SENSOR_ALIGN_DIRECTION_REVERSE
} SensorAlignDirection;

typedef enum
{
    SENSOR_ALIGN_REFERENCE_NONE,
    SENSOR_ALIGN_REFERENCE_SLOT
} SensorAlignReferenceType;


typedef k32s SensorAlignSelectType;

#define SENSOR_ALIGN_SELECT_MID     (0)
#define SENSOR_ALIGN_SELECT_WIDEST  (1)

typedef k32s GvAlignDegreesOfFreedom;

#define GV_ALIGN_DOF_NONE       (0x00)
#define GV_ALIGN_DOF_X          (0x20)
#define GV_ALIGN_DOF_Y          (0x10)
#define GV_ALIGN_DOF_Z          (0x08)
#define GV_ALIGN_DOF_A          (0x04)
#define GV_ALIGN_DOF_B          (0x02)
#define GV_ALIGN_DOF_C          (0x01)
#define GV_ALIGN_DOF_Z_B        (0x0A)
#define GV_ALIGN_DOF_XZ_B       (0x2A)
#define GV_ALIGN_DOF_YZ_B       (0x1A)
#define GV_ALIGN_DOF_XYZ_B      (0x3A)
#define GV_ALIGN_DOF_YZ_BC      (0x1B)
#define GV_ALIGN_DOF_XYZ_BC     (0x3B)
#define GV_ALIGN_DOF_YZ_ABC     (0x1F)
#define GV_ALIGN_DOF_XYZ_ABC    (0x3F)

typedef k32s SensorAreaType;

#define SENSOR_AREA_OBJECT          (0)
#define SENSOR_AREA_CLEARANCE       (1)
#define SENSOR_AREA_OBJ_PLUS_CLEAR  (2)
#define SENSOR_AREA_OBJ_MINUS_CLEAR (3)

typedef k32s GvProfilePanelEdgeType;
#define GV_PROFILE_PANEL_EDGE_BLOCK_TANGENT_EDGE    (0)
#define GV_PROFILE_PANEL_EDGE_BLOCK_CORNER_EDGE     (1)
#define GV_PROFILE_PANEL_BLOCK_TANGENT_EDGE         (0)
#define GV_PROFILE_PANEL_BLOCK_CORNER_EDGE          (1)


typedef k32s SensorAreaBaseline;

#define SENSOR_AREA_BASELINE_X_AXIS (0)
#define SENSOR_AREA_BASELINE_LINE   (1)

typedef k32s SensorProfileFeatureType;
#define SENSOR_PROFILE_FEATURE_TOP              (0)
#define SENSOR_PROFILE_FEATURE_BOTTOM           (1)
#define SENSOR_PROFILE_FEATURE_RIGHT            (2)
#define SENSOR_PROFILE_FEATURE_LEFT             (3)
#define SENSOR_PROFILE_FEATURE_CORNER           (4)
#define SENSOR_PROFILE_FEATURE_AVERAGE          (5)
#define SENSOR_PROFILE_FEATURE_EDGE_RISING      (6)
#define SENSOR_PROFILE_FEATURE_EDGE_FALLING     (7)
#define SENSOR_PROFILE_FEATURE_EDGE             (8)
#define SENSOR_PROFILE_FEATURE_CORNER_TOP       (9)
#define SENSOR_PROFILE_FEATURE_CORNER_BOTTOM    (10)
#define SENSOR_PROFILE_FEATURE_CORNER_LEFT      (11)
#define SENSOR_PROFILE_FEATURE_CORNER_RIGHT     (12)
#define SENSOR_PROFILE_FEATURE_MEDIAN           (13)
#define SENSOR_PROFILE_FEATURE_TYPE_COUNT       (SENSOR_PROFILE_FEATURE_MEDIAN + 1)

typedef k32s GvProfilePanelReference;
#define GV_PROFILE_PANEL_BLOCK_REFERENCE_LEFT_EDGE      (0)
#define GV_PROFILE_PANEL_BLOCK_REFERENCE_RIGHT_EDGE     (1)
#define GV_PROFILE_PANEL_BLOCK_REFERENCE_LEFT_SURFACE   (2)
#define GV_PROFILE_PANEL_BLOCK_REFERENCE_RIGHT_SURFACE  (3)
#define GV_PROFILE_PANEL_BLOCK_REFERENCE_LEFT_DISTANCE  (4)
#define GV_PROFILE_PANEL_BLOCK_REFERENCE_RIGHT_DISTANCE (5)

typedef enum
{
    SENSOR_PROFILE_EDGE_RISING  = 0x1,
    SENSOR_PROFILE_EDGE_FALLING = 0x2
} SensorProfileEdgeType;

typedef enum
{
    SENSOR_PROFILE_CORNER_TOP  = 0x1,
    SENSOR_PROFILE_CORNER_BOTTOM = 0x2,
    SENSOR_PROFILE_CORNER_LEFT = 0x4,
    SENSOR_PROFILE_CORNER_RIGHT = 0x8
} SensorProfileCornerType;

typedef k32s SensorProfileBaselineType;

#define PROFILE_BASELINE_TYPE_X_AXIS        (0)
#define PROFILE_BASELINE_TYPE_Z_AXIS        (1)
#define PROFILE_BASELINE_TYPE_FIT_LINE      (2)
#define PROFILE_BASELINE_TYPE_TWO_POINTS    (3)

typedef enum
{
    GOCATOR_CALIBRATION_NO_OBJECT            = -1,
    GOCATOR_CALIBRATION_NO_TARGET            = -2,
    GOCATOR_CALIBRATION_TARGET_CONFIG_ERROR  = -3,
    GOCATOR_CALIBRATION_TARGET_POS_ERROR     = -4,
    GOCATOR_CALIBRATION_NO_REFERENCE_SLOT    = -5,
    GOCATOR_CALIBRATION_NO_TRAVEL_DETECTED   = -6,
    GOCATOR_CALIBRATION_TOO_FEW_PROFILES     = -7
} GocatorCalibrationStatus;

typedef struct GvProfileAlignEntry
{
    kPoint3d64f angle;
    kPoint3d64f position;
    k64f time;
    k64s count;
    k64u frameCount;
    k64u framesBuffered;
    k64u framesExpected;
} GvProfileAlignEntry;

typedef enum
{
    GOCATOR_ALIGNMENT_REPORT_ID_NONE = 0,
    GOCATOR_ALIGNMENT_REPORT_ID_BUFFERRED_BAR
} GocatorAlignmentReportId;

#define GV_PROFILE_ALIGN_MAX_SENSORS        (32)
typedef struct GvProfileAlignOutput
{
    k64f encoderResolution;
    k64f speed;
    k32u entryCount;
    GvProfileAlignEntry entry[GV_PROFILE_ALIGN_MAX_SENSORS];
    k32u outputToClientId;  // id of output information to send to client, 0 is nothing to send
} GvProfileAlignOutput;

kDeclareValueEx(Gv, GvProfileAlignOutput, kValue)

typedef enum
{
    GOCATOR_ALIGNMENT_ERROR_MSG_ID_NONE = 0,
    GOCATOR_ALIGNMENT_ERROR_MSG_ID_BUFFERRED_BAR_TOO_FEW1,
    GOCATOR_ALIGNMENT_ERROR_MSG_ID_BUFFERRED_BAR_TOO_FEW2
} GocatorAlignmentErrorMessageId;

#define GV_PROFILE_ALIGN_STATUS_MAX_PARAMS  (3)
typedef struct GvProfileAlignStatus
{
    k32s status;
    k32s errorMsgId;    // of type GocatorAlignmentErrorMessageId
    k32s paramCount;
    k32s params[GV_PROFILE_ALIGN_STATUS_MAX_PARAMS];
} GvProfileAlignStatus;

kDeclareValueEx(Gv, GvProfileAlignStatus, kValue)

typedef enum
{
    GV_SURFACE_RIVET_FIND_TYPE_HOLE,
    GV_SURFACE_RIVET_FIND_TYPE_BUMP,
    GV_SURFACE_RIVET_FIND_TYPE_FLUSH
} GvSurfaceRivetFindType;

typedef enum
{
    GV_SURFACE_RIVET_HEIGHT_TYPE_NONE      = 0,
    GV_SURFACE_RIVET_HEIGHT_TYPE_NOMINAL,
    GV_SURFACE_RIVET_HEIGHT_TYPE_MAX,
    GV_SURFACE_RIVET_HEIGHT_TYPE_MEAN,
    GV_SURFACE_RIVET_HEIGHT_TYPE_FIT
} GvSurfaceRivetHeightType;

typedef enum
{
    GV_SURFACE_RIVET_MEASUREMENT_X,
    GV_SURFACE_RIVET_MEASUREMENT_Y,
    GV_SURFACE_RIVET_MEASUREMENT_Z,
    GV_SURFACE_RIVET_MEASUREMENT_RADIUS,
    GV_SURFACE_RIVET_MEASUREMENT_X_ANGLE,
    GV_SURFACE_RIVET_MEASUREMENT_Y_ANGLE,
    GV_SURFACE_RIVET_MEASUREMENT_TILT_ANGLE,
    GV_SURFACE_RIVET_MEASUREMENT_TILT_DIRECTION,
    GV_SURFACE_RIVET_MEASUREMENT_TOP_MIN,
    GV_SURFACE_RIVET_MEASUREMENT_TOP_MAX,
    GV_SURFACE_RIVET_MEASUREMENT_TOP_MEAN,
    GV_SURFACE_RIVET_MEASUREMENT_TOP_STDEV,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_OFFSET_MIN,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_OFFSET_MAX,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_OFFSET_MEAN,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_OFFSET_STDEV,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_SLOPE_MIN,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_SLOPE_MAX,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_SLOPE_MEAN,
    GV_SURFACE_RIVET_MEASUREMENT_RADIAL_SLOPE_STDEV
} GvSurfaceRivetMeasureType;

typedef k32s GvSurfaceFrameOfReferenceType;
#define GV_SURFACE_UTILITIES_FOR_TYPE_SENSOR     (0)
#define GV_SURFACE_UTILITIES_FOR_TYPE_SCAN       (1)
#define GV_SURFACE_UTILITIES_FOR_TYPE_SOURCEPART (3)
#define GV_SURFACE_UTILITIES_FOR_TYPE_PART       (2)

typedef enum
{
    GV_SURFACE_RADIAL_SCAN_TYPE_RADIAL,
    GV_SURFACE_RADIAL_SCAN_TYPE_ROW,
    GV_SURFACE_RADIAL_SCAN_TYPE_COLUMN,
    GV_SURFACE_RADIAL_SCAN_TYPE_ROWCOL
} GvSurfaceRadialScanType;

typedef enum
{
    GV_PART_THRESHOLD_TYPE_ABOVE,
    GV_PART_THRESHOLD_TYPE_BELOW,
    GV_PART_THRESHOLD_TYPE_RANGE
} GvPartThresholdType;

#define GV_PART_MATCH_DEFAULT_MAX_EDGES               (4096)

typedef enum
{
    GV_SURFACE_RADIAL_DROP_OUT_INVALID_COUNT,
    GV_SURFACE_RADIAL_DROP_OUT_VALID_COUNT,
    GV_SURFACE_RADIAL_DROP_OUT_INVALID_AREA,
    GV_SURFACE_RADIAL_DROP_OUT_VALID_AREA
} GvSurfaceRadialDropOutType;

typedef enum
{
    GV_RUN_CONTROL_FREE_RUN,
    GV_RUN_CONTROL_INPUT_START_PULSE,
    GV_RUN_CONTROL_INPUT_LEVEL
} GvRunControlType;

typedef enum
{
    // OK means there is a usable/valid measurement value to work with.
    GV_MEASUREMENT_OK = 0,
    // Measurement error is treated as a bit mask.
    // Reserve bit value 0x1 for upper layer software use to indicate to clients
    // the measurement value FAILED, meaning it lies outside a client
    // specified decision min/max range.
    GV_MEASUREMENT_ERROR_FAILED         = 0x1,  
    GV_MEASUREMENT_ERROR_INVALID_RESULT = 0x2,  // Could not measure a value.
    GV_MEASUREMENT_ERROR_INVALID_ANCHOR = 0x4,
    GV_MEASUREMENT_ERROR_PART_MATCH     = 0x8
} GvMeasurementError;

typedef enum GvProfileFilterType
{
    GV_PROFILE_FILTER_TYPE_MEAN,
    GV_PROFILE_FILTER_TYPE_MEDIAN,
    GV_PROFILE_FILTER_TYPE_DIFFERENCE,
    GV_PROFILE_FILTER_TYPE_NONE
} GvProfileFilterType;

typedef enum GvProfileFilterDimension
{
    GV_PROFILE_FILTER_DIMENSION_X,
    GV_PROFILE_FILTER_DIMENSION_Y
} GvProfileFilterDimension;

typedef struct GvPartEdgePoint2d
{
    k64f x;
    k64f y;
    k64f direction;
} GvPartEdgePoint2d;

typedef struct kView1
{
    k32u start;
    k32u count;
    k32s step;
} kView1;

#define kVIEW1(VIEW, START, COUNT, STEP)                                        \
    ((VIEW)->start = (START), (VIEW)->count = (COUNT), (VIEW)->step = (STEP),   \
     (VIEW))

typedef struct kView2
{
    k32u start[2];
    k32u count[2];
    k32s step[2];
} kView2;

#define kVIEW2(VIEW, START0, START1, COUNT0, COUNT1, STEP0, STEP1)              \
    ((VIEW)->start[0] = (START0), (VIEW)->start[1] = (START1),                  \
     (VIEW)->count[0] = (COUNT0), (VIEW)->count[1] = (COUNT1),                  \
     (VIEW)->step[0]  = (STEP0),  (VIEW)->step[1]  = (STEP1),                   \
     (VIEW))

#define GV_SURFACE_CLOCK_TO_S(C)        ((C) / 1024000.0) // there must be a macro for this
#define GV_SURFACE_S_TO_CLOCK(C)        ((C) * 1024000)

// Health Indicators - public unless otherwise specified

#define GV_INDICATOR_ROTATIONAL_GENERATION_DROPS    (22000)
#define GV_INDICATOR_COMBINER_DROPS                 (22001) // private
#define GV_INDICATOR_STRIPE_DROPS                   (22002) // private
#define GV_INDICATOR_SORTER_DROPS                   (22003) // private
#define GV_INDICATOR_RUNTIME                        (22004)
#define GV_INDICATOR_PHASE_DROPS                    (22005) // private
// part detector indicators
#define GV_INDICATOR_PART_EMITTED                   (22006)
#define GV_INDICATOR_PART_LENGTH_LIMIT              (22007)
#define GV_INDICATOR_PART_AREA_DROPS                (22008)
#define GV_INDICATOR_PART_BACKTRACK_DROPS           (22009)
#define GV_INDICATOR_PART_ACTIVE                    (22010)
#define GV_INDICATOR_PART_LENGTH                    (22011)
#define GV_INDICATOR_PART_START_Y                   (22012)
#define GV_INDICATOR_PART_TRACKING_STATE            (22013)
#define GV_INDICATOR_PART_CAPACITY                  (22014)
#define GV_INDICATOR_PART_X_POSITION                (22015)
// runtime indicators
#define GV_INDICATOR_RUNTIME_MIN                    (22016)
#define GV_INDICATOR_RUNTIME_MAX                    (22017)
#define GV_INDICATOR_RUNTIME_AVG                    (22018)
#define GV_INDICATOR_RUNPERCENT_AVG                 (22019)
// alignment indicator(s)
#define GV_INDICATOR_BAR_ALIGNMENT_STATUS           (22020)


typedef struct GvTracheidMoment
{
    k32s moments[6];
} GvTracheidMoment;


#define GV_THRESHOLD_EQUALS(A, B, EPSILON) ((kMath_Max_(A, B) - kMath_Min_(A, B)) < EPSILON)
#define GV_THRESHOLD_GREATER(A, B, EPSILON) (!GV_THRESHOLD_EQUALS(A,B,EPSILON) && A > B)
#define GV_THRESHOLD_LESS(A, B, EPSILON) (!GV_THRESHOLD_EQUALS(A,B,EPSILON) && A < B)
#define GV_THRESHOLD_GREATER_OR_EQUAL(A, B, EPSILON) (GV_THRESHOLD_EQUALS(A,B,EPSILON) || A > B)
#define GV_THRESHOLD_LESS_OR_EQUAL(A, B, EPSILON) (GV_THRESHOLD_EQUALS(A,B,EPSILON) || A < B)

kDeclareValueEx(Gv, GvTracheidMoment, kValue)

/**
* @struct  GdkMeasurementDecision
* @extends kValue
* @ingroup Gv
* @brief   Represents a measurement result decision.
* Note:- This is added here to remove dependancy of Govision over Gdk
*/
typedef k8u GdkMeasurementDecision;

/** @relates GdkMeasurementDecision @{ */
#define GDK_MEASUREMENT_OK               (0)         ///< No error.
#define GDK_MEASUREMENT_ERROR_VALUE      (0x2)       ///< Invalid measurement value.
#define GDK_MEASUREMENT_ERROR_ANCHOR     (0x4)       ///< Invalid anchor input.
#define GDK_MEASUREMENT_ERROR_PART_MATCH (0x8)       ///< Part matching failure.
/** @} */
#endif
