/**
 * @file    TestSurfaceGraphics.c
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestSurfaceGraphics.h>

kBeginClassEx(Tool, TestSurfaceGraphics)
    kAddVMethod(TestSurfaceGraphics, kObject, VRelease)
    kAddVMethod(TestSurfaceGraphics, GdkTool, VInit)
    kAddVMethod(TestSurfaceGraphics, GdkTool, VDescribe)
    kAddVMethod(TestSurfaceGraphics, GdkTool, VUpdateConfig)
    kAddVMethod(TestSurfaceGraphics, GdkTool, VStart)
    kAddVMethod(TestSurfaceGraphics, GdkTool, VStop)
    kAddVMethod(TestSurfaceGraphics, GdkTool, VProcess)
kEndClassEx()

ToolFx(kStatus) TestSurfaceGraphics_VDescribe(GdkToolInfo toolInfo)
{
    GdkMeasurementInfo mmtInfo;
    GdkRegion3d64f defRegion = { -4, -3, -2, 2, 2, 4, 0 };
    k64f defFloat1 = 1.0, defFloat2 = 2.0;

    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_SURFACE_GRAPHICS_TOOL_NAME));
    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_SURFACE_GRAPHICS_TOOL_LABEL));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_X, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Y, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Z, kTRUE));

    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_SURFACE_REGION,    "Region",   "Region",       &defRegion, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT,             "StartSize","Start Size",   &defFloat1, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_FLOAT,             "StepSize", "Step Size",    &defFloat2, kNULL));

    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Point", "Point", &mmtInfo));
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Line",  "Line",  &mmtInfo));

    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_VInit(TestSurfaceGraphics tool, kType type, kAlloc alloc)
{
    kObjR(TestSurfaceGraphics, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    kZero(obj->region);
    obj->startSize = 0.0f;
    obj->stepSize = 0.0f;

    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_VRelease(TestSurfaceGraphics tool)
{
    kObj(TestSurfaceGraphics, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestSurfaceGraphics_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_VStart(TestSurfaceGraphics tool)
{
    kObj(TestSurfaceGraphics, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);
    GdkParam param;

    obj->dataSource = GdkToolCfg_Source(config);

    param = GdkParams_Find(params, "Region");
    obj->region = *GdkParam_AsSurfaceRegion(param);

    param = GdkParams_Find(params, "StartSize");
    obj->startSize = (k32f) GdkParam_AsFloat(param);

    param = GdkParams_Find(params, "StepSize");
    obj->stepSize = (k32f)GdkParam_AsFloat(param);

    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_VStop(TestSurfaceGraphics tool)
{
    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_OutputPoints(TestSurfaceGraphics tool, GdkRegion3d64f *region, GdkToolOutput output)
{
    kObj(TestSurfaceGraphics, tool);
    GdkGraphic graphic = kNULL;
    GdkGraphicPointSet pointSet = kNULL;
    kStatus exception = kOK;
    kAlloc alloc;

    alloc = GdkTool_MessageAlloc(tool);

    kTry
    {
        kPoint3d32f point32f; // Note, we need to use kPoint3d32f, NOT 64f!

        kColor colors[5] = { kCOLOR_RED, kCOLOR_GREEN, kCOLOR_BLUE, kCOLOR_YELLOW, kCOLOR_ORANGE };
        kMarkerShape shapes[6] = { kMARKER_SHAPE_CROSS, kMARKER_SHAPE_PLUS, kMARKER_SHAPE_CIRCLE, kMARKER_SHAPE_SQUARE, kMARKER_SHAPE_TRIANGLE, kMARKER_SHAPE_DIAMOND };

        k32u colorIdx, shapeIdx;

        k32f xStep = (k32f)region->width / (5 + 2); // We draw 5 sizes across X, and leave one margin on each side
        k32f yStep = (k32f)region->length / (6 + 2); // We draw 6 shapes across X and leave one margin on each side
        k32f size;

        kTest(GdkGraphic_Construct(&graphic, alloc));


        point32f.y = (k32f)region->y + yStep;
        point32f.z = (k32f)(region->z + region->height / 2.0);

        for (shapeIdx = 0; shapeIdx < 6; shapeIdx++)
        {
            size = obj->startSize;
            point32f.x = (k32f)region->x + xStep; // start with one step into the region
            for (colorIdx = 0; colorIdx < 5; colorIdx++)
            {
                // Note. You can also do multiple points in one function. They will all have the same size, color and shape.
                kTest(GdkGraphicPointSet_Construct(&pointSet, size, shapes[shapeIdx], colors[colorIdx], &point32f, 1, alloc));
                kTest(GdkGraphic_AddPointSet(graphic, pointSet));
                pointSet = kNULL;
                point32f.x += xStep;
                size += obj->stepSize;
            }
            point32f.y += yStep;
        }

        kTest(GdkToolOutput_SetRendering(output, TEST_SURFACE_GRAPHICS_MEASUREMENT_POINT, graphic));
        graphic = kNULL;
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kDestroyRef(&pointSet);
        kDestroyRef(&graphic);
        kEndFinallyEx();
    }
    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_OutputLines(TestSurfaceGraphics tool, GdkRegion3d64f *region, GdkToolOutput output)
{
    kObj(TestSurfaceGraphics, tool);
    GdkGraphic graphic = kNULL;
    GdkGraphicLineSet lineSet = kNULL;
    kStatus exception = kOK;
    kAlloc alloc;

    alloc = GdkTool_MessageAlloc(tool);

    kTry
    {
        kPoint3d32f point32f[2]; // Note, we need to use kPoint3d32f, NOT 64f!
        kColor colors[5] = { kCOLOR_RED, kCOLOR_GREEN, kCOLOR_BLUE, kCOLOR_YELLOW, kCOLOR_ORANGE };

        k32u colorIdx;
        k32f xStep = (k32f)region->width / (5 + 2); // We draw 5 color across region
        k32f yStep = (k32f)region->length / (5 + 2); // We draw 5 sizes across X and leave one margin on each side
        k32f size;
        k32u i;

        kTest(GdkGraphic_Construct(&graphic, alloc));

        point32f[0].y = (k32f)region->y;
        point32f[1].y = (k32f)region->y;
        point32f[0].z = point32f[1].z = (k32f)(region->z + region->height / 2.0);

        size = obj->startSize;

        for (i = 0; i < 5; i++)
        {
            point32f[0].x = (k32f)region->x; // start with one step into the region
            point32f[1].x = (k32f)region->x + xStep;
            for (colorIdx = 0; colorIdx < 5; colorIdx++)
            {
                // Note. You can also do multiple line points to draw a polygon. Lines are connected between element in the order it is passed in
                kTest(GdkGraphicLineSet_Construct(&lineSet, size, colors[colorIdx], point32f, 2, alloc));
                kTest(GdkGraphic_AddLineSet(graphic, lineSet));
                lineSet = kNULL;
                point32f[0].x += xStep;
                point32f[1].x += xStep;
            }
            point32f[0].y += yStep;
            point32f[1].y += yStep;
            size += obj->stepSize;
        }

        kTest(GdkToolOutput_SetRendering(output, TEST_SURFACE_GRAPHICS_MEASUREMENT_LINE, graphic));
        graphic = kNULL;
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kDestroyRef(&lineSet);
        kDestroyRef(&graphic);
        kEndFinallyEx();
    }
    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_VProcess(TestSurfaceGraphics tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestSurfaceGraphics, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkInputItem item;
    const kPoint3d64f* anchor = GdkToolInput_AnchorPosition(input);
    GdkRegion3d64f offsetRegion = { k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL };
    kStatus exception = kOK;

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item) return kERROR_PARAMETER;

    kTry
    {
        // Adjust user region based on anchoring information. If anchoring is not enabled, anchor is zero.
        offsetRegion = obj->region;
        offsetRegion.x += anchor->x;
        offsetRegion.y += anchor->y;
        offsetRegion.z += anchor->z;

        if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, TEST_SURFACE_GRAPHICS_MEASUREMENT_POINT)))
        {
            kTest(TestSurfaceGraphics_OutputPoints(tool, &offsetRegion, output));
            kTest(TestSurfaceGraphics_OutputValue(output, TEST_SURFACE_GRAPHICS_MEASUREMENT_POINT, 0.0, GDK_MEASUREMENT_OK, config));
        }

        if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, TEST_SURFACE_GRAPHICS_MEASUREMENT_LINE)))
        {
            kTest(TestSurfaceGraphics_OutputLines(tool, &offsetRegion, output));
            kTest(TestSurfaceGraphics_OutputValue(output, TEST_SURFACE_GRAPHICS_MEASUREMENT_LINE, 0.0, GDK_MEASUREMENT_OK, config));
        }
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kEndFinallyEx();
    }

    return kOK;
}

ToolFx(kStatus) TestSurfaceGraphics_OutputValue(GdkToolOutput output, kSize index, k64f value, GdkMeasurementDecision decision, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvMeasureMsg_SetValue(msg, value));
            kCheck(GvMeasureMsg_SetStatus(msg, decision));
        }
    }

    return kOK;
}
