/**
 * @file    TestFeatureAverage.c
 * @brief   Tool to compute the average of two features (points or lines)
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestFeatureAverage.h>
#include <kVision/Common/kNullMath.h>
#include <math.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

kBeginClassEx(Tool, TestFeatureAverage)
    kAddVMethod(TestFeatureAverage, kObject, VRelease)
    kAddVMethod(TestFeatureAverage, GdkTool, VInit)
    kAddVMethod(TestFeatureAverage, GdkTool, VName)
    kAddVMethod(TestFeatureAverage, GdkTool, VDescribe)
    kAddVMethod(TestFeatureAverage, GdkTool, VNewMeasurementConfig)
    kAddVMethod(TestFeatureAverage, GdkTool, VUpdateConfig)
    kAddVMethod(TestFeatureAverage, GdkTool, VNewToolConfig)
    kAddVMethod(TestFeatureAverage, GdkTool, VStart)
    kAddVMethod(TestFeatureAverage, GdkTool, VStop)
    kAddVMethod(TestFeatureAverage, GdkTool, VProcess)
kEndClassEx()

/////////////////////////////////////////////////////////////////////////////
// GtTool functions
/////////////////////////////////////////////////////////////////////////////

ToolFx(const kChar*) TestFeatureAverage_VName()
{
 return GDK_SURFACE_LINE_AVERAGE_TOOL_NAME;
}

ToolFx(kStatus) TestFeatureAverage_VDescribe(GdkToolInfo toolInfo)
{
    GdkParamInfo paramInfo = kNULL;
    GdkMeasurementInfo mmtInfo;
    GdkFeatureInfo featureInfo = kNULL;

    kCheck(GdkToolInfo_SetLabel(toolInfo, GDK_SURFACE_LINE_AVERAGE_TOOL_LABEL));
    kCheck(GdkToolInfo_SetTypeName(toolInfo, GDK_SURFACE_LINE_AVERAGE_TOOL_NAME));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_SURFACE));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    // Input Features
    // Add two input features, each could be either a line or points
    kCheck(GdkToolInfo_AddInput(toolInfo, GDK_DATA_TYPE_FEATURE_LINE, "Feature0", "Feature 0", &paramInfo));
    GdkParamInfo_AddDataType(paramInfo, GDK_DATA_TYPE_FEATURE_POINT);
    kCheck(GdkToolInfo_AddInput(toolInfo, GDK_DATA_TYPE_FEATURE_LINE, "Feature1", "Feature 1", &paramInfo));
    GdkParamInfo_AddDataType(paramInfo, GDK_DATA_TYPE_FEATURE_POINT);

    // Output measurements
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "X", "X", &mmtInfo));                         // #0
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_X));
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Y", "Y", &mmtInfo));                         // #1
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Y));
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Z", "Z", &mmtInfo));                         // #2
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Z));
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "ZAngle", "ZAngle", &mmtInfo));               // #3
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Z_ANGLE));

    // Ouput Features
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_FEATURE_POINT, "PointOut", "Point Output", &featureInfo)); // #4
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_FEATURE_LINE,  "LineOut",  "Line Output",  &featureInfo)); // #5

    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VInit(TestFeatureAverage tool, kType type, kAlloc alloc)
{
    kObjR(TestFeatureAverage, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);

    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VRelease(TestFeatureAverage tool)
{
    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestFeatureAverage_VNewToolConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VNewMeasurementConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkMeasurementCfg measurementConfig)
{
    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VStart(TestFeatureAverage tool)
{
    kObj(TestFeatureAverage, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    obj->dataSource = GdkToolCfg_Source(config);

    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VStop(TestFeatureAverage tool)
{
    kObj(TestFeatureAverage, tool);
    return kOK;
}

ToolFx(kStatus) TestFeatureAverage_VProcess(TestFeatureAverage tool, GdkToolInput input, GdkToolOutput output)
{
    kStatus status = kOK;
    kSize i = 0;
    kObj(TestFeatureAverage, tool);
    GdkSurfaceInput item = kNULL;
    GdkDataInfo itemInfo = kNULL;
    const GdkRegion3d64f* globalRegion = kNULL;

    GdkToolCfg config = GdkTool_Config(tool);
    GdkFeature feat0 = kNULL;
    GdkFeature feat1 = kNULL;
    const kPoint3d64f *pt1 = kNULL, *pt2 = kNULL;
    kLine3d64f line1 = {0,0}, line2 = {0,0};
    k64f zAngle = 0;
    kLine3d64f avgLine = { {k64F_NULL,k64F_NULL,k64F_NULL},{k64F_NULL,k64F_NULL,k64F_NULL}};
    kPoint3d64f avgPoint = { k64F_NULL,k64F_NULL,k64F_NULL };
    GdkGraphic graphic = kNULL;

    kAlloc alloc = GdkTool_MessageAlloc(tool);

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item)
    {
        return kERROR_PARAMETER;
    }
    itemInfo = GdkInputItem_Info(item);
    globalRegion = GdkDataInfo_Region(itemInfo);

    // Obtain the input features
    feat0 = GdkToolInput_FeatureAt(input, 0);
    feat1 = GdkToolInput_FeatureAt(input, 1);

    if (feat0 == kNULL || feat1 == kNULL)
    {
        return kOK;
    }

    kTry
    {
        // Create the graphics object
        GdkGraphic_Construct(&graphic, alloc);

        // Act according to the features' types
        if (GdkFeature_Type(feat0) == GDK_FEATURE_TYPE_LINE &&
            GdkFeature_Type(feat1) == GDK_FEATURE_TYPE_LINE)
        {
            line1.p = *GdkLineFeature_Point(feat0);
            kTest(GdkLineFeature_Direction(feat0, &line1.dir));
            line2.p = *GdkLineFeature_Point(feat1);
            kTest(GdkLineFeature_Direction(feat1, &line2.dir));

            kTest(TestFeatureAverage_AverageLineLine(line1, line2, &avgLine));
            avgPoint = avgLine.p;

            // Draw the features to the graphics object
            kTest(TestFeatureAverage_DrawLine( graphic, output, &line1,    kCOLOR_YELLOW, globalRegion, alloc));
            kTest(TestFeatureAverage_DrawLine( graphic, output, &line2,    kCOLOR_YELLOW, globalRegion, alloc));
            kTest(TestFeatureAverage_DrawLine( graphic, output, &avgLine,  kCOLOR_BLUE,   globalRegion, alloc));
        }
        else if ((GdkFeature_Type(feat0) == GDK_FEATURE_TYPE_POINT && GdkFeature_Type(feat1) == GDK_FEATURE_TYPE_LINE) ||
                 (GdkFeature_Type(feat0) == GDK_FEATURE_TYPE_LINE  && GdkFeature_Type(feat1) == GDK_FEATURE_TYPE_POINT))
        {
            if (GdkFeature_Type(feat0) == GDK_FEATURE_TYPE_POINT)
            {
                pt1 = GdkPointFeature_Position(feat0);
                line1.p = *GdkLineFeature_Point(feat1);
                kTest(GdkLineFeature_Direction(feat1, &line1.dir));
            }
            else
            {
                pt1 = GdkPointFeature_Position(feat1);
                line1.p = *GdkLineFeature_Point(feat0);
                kTest(GdkLineFeature_Direction(feat0, &line1.dir));
            }

            kTest(TestFeatureAverage_AveragePointLine(*pt1, line1, &avgPoint));

            // Draw the features to the graphics object
            kTest(TestFeatureAverage_DrawPoint(graphic, output, pt1,       kCOLOR_YELLOW,               alloc));
            kTest(TestFeatureAverage_DrawLine( graphic, output, &line1,    kCOLOR_YELLOW, globalRegion, alloc));
            kTest(TestFeatureAverage_DrawPoint(graphic, output, &avgPoint, kCOLOR_BLUE,                 alloc));
        }
        else if (GdkFeature_Type(feat0) == GDK_FEATURE_TYPE_POINT &&
            GdkFeature_Type(feat1) == GDK_FEATURE_TYPE_POINT)
        {
            pt1 = GdkPointFeature_Position(feat0);
            pt2 = GdkPointFeature_Position(feat1);

            kTest(TestFeatureAverage_AveragePointPoint(*pt1, *pt2, &avgPoint));

            // Draw the features to the graphics object
            kTest(TestFeatureAverage_DrawPoint(graphic, output, pt1,       kCOLOR_YELLOW,               alloc));
            kTest(TestFeatureAverage_DrawPoint(graphic, output, pt2,       kCOLOR_YELLOW,               alloc));
            kTest(TestFeatureAverage_DrawPoint(graphic, output, &avgPoint, kCOLOR_BLUE,                 alloc));
        }

        // Output Feature
        kTest(TestFeatureAverage_SendFeaturePoint(tool, output, OUTPUT_FEATURE_INDEX_POINT, &avgPoint));
        kTest(TestFeatureAverage_SendFeatureLine( tool, output, OUTPUT_FEATURE_INDEX_LINE,  &avgLine));

        kTest(GdkToolOutput_SetRendering(output, 0, graphic));
        graphic = kNULL;

        // Calculate the line angle
        if (avgLine.dir.x != k64F_NULL)
        {
            zAngle = kMath_RadToDeg_(atan2(avgLine.dir.y, avgLine.dir.x));
        }

        // Ouput Measurements
        kTest(TestFeatureAverage_OutputValue(output, OUTPUT_MEASURE_INDEX_X, avgPoint.x, GDK_MEASUREMENT_OK, config));
        kTest(TestFeatureAverage_OutputValue(output, OUTPUT_MEASURE_INDEX_Y, avgPoint.y, GDK_MEASUREMENT_OK, config));
        kTest(TestFeatureAverage_OutputValue(output, OUTPUT_MEASURE_INDEX_Z, avgPoint.z, GDK_MEASUREMENT_OK, config));
        kTest(TestFeatureAverage_OutputValue(output, OUTPUT_MEASURE_INDEX_ZAngle, zAngle, zAngle != k64F_NULL? GDK_MEASUREMENT_OK: GDK_MEASUREMENT_ERROR_VALUE, config));
    }
    kCatchEx(&status)
    {
        for (i = 0; i < GdkToolCfg_MeasurementCount(config); i++)
        {
            TestFeatureAverage_OutputValue(output, i, k64F_NULL, GDK_MEASUREMENT_ERROR_VALUE, config);
        }
        kEndCatchEx(kOK);
    }
    kFinallyEx
    {
        kDestroyRef(&graphic);
        kEndFinallyEx();
    }

    return kOK;
}

/////////////////////////////////////////////////////////////////////////////
// Output rendering
/////////////////////////////////////////////////////////////////////////////

// Output a measurement value to a certain index
ToolFx(kStatus) TestFeatureAverage_OutputValue(GdkToolOutput output, kSize index, k64f value, GdkMeasurementDecision decision, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvMeasureMsg_SetValue(msg, value));
            kCheck(GvMeasureMsg_SetStatus(msg, decision));
        }
    }

    return kOK;
}

// Copy the feature's properties to the output feature
ToolFx(kStatus) TestFeatureAverage_SendFeatureLine(TestFeatureAverage tool, GdkToolOutput output, kSize index, const kLine3d64f* inLine)
{
    GvLineFeatureMsg msg = kNULL;

    if (inLine->p.x != k64F_NULL && inLine->p.y != k64F_NULL && inLine->p.z != k64F_NULL)
    {
        // Get the output feature
        kCheck(GdkToolOutput_InitFeatureAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvFeatureMsg_SetPosition(msg, &inLine->p));
            kCheck(GvLineFeatureMsg_SetDirection(msg, &inLine->dir));
        }
    }

    return kOK;
}
ToolFx(kStatus) TestFeatureAverage_SendFeaturePoint(TestFeatureAverage tool, GdkToolOutput output, kSize index, const kPoint3d64f* inPt)
{
    GvPointFeatureMsg msg = kNULL;

    // Get the output feature
    kCheck(GdkToolOutput_InitFeatureAt(output, index, &msg));
    if (msg != kNULL)
    {
        kCheck(GvFeatureMsg_SetPosition(msg, inPt));
    }

    return kOK;
}

// Draw a point to the canvas
ToolFx(kStatus) TestFeatureAverage_DrawPoint(GdkGraphic graphic, GdkToolOutput output, const kPoint3d64f* pt, kColor color, kAlloc alloc)
{
    kPoint3d32f point32f = {0};
    GdkGraphicPointSet pointSet = kNULL;

    kTry
    {
        kPoint3d_Init_(&point32f, (k32f)pt->x, (k32f)pt->y, (k32f)pt->z);
        kTest(GdkGraphicPointSet_Construct(&pointSet, 3.0, kMARKER_SHAPE_DIAMOND, color, &point32f, 1, alloc));
        kTest(GdkGraphic_AddPointSet(graphic, pointSet));
        pointSet = kNULL;
    }
    kFinally
    {
        kDestroyRef(&pointSet);
        kEndFinally();
    }

    return kOK;

}

// Draw a line to the canvas
ToolFx(kStatus) TestFeatureAverage_DrawLine(GdkGraphic graphic, GdkToolOutput output, const kLine3d64f* line, kColor color, const GdkRegion3d64f* globalRegion, kAlloc alloc)
{
    GdkGraphicLineSet lineSet = kNULL;
    kPoint3d32f points[2] = { { 0, 0, 0 }, { 0, 0, 0 } };

    kTry
    {
        // Create two points to make the line. lines go from minimum Y to maximum Y
        if (line->dir.y != 0)
        {
            points[0] = TestFeatureAverage_Point64to32(TestFeatureAverage_LineSolveForY(*line, globalRegion->y));
            points[1] = TestFeatureAverage_Point64to32(TestFeatureAverage_LineSolveForY(*line, globalRegion->y + globalRegion->length));
        }
        else
        {
            points[0] = TestFeatureAverage_Point64to32(TestFeatureAverage_LineSolveForX(*line, globalRegion->x));
            points[1] = TestFeatureAverage_Point64to32(TestFeatureAverage_LineSolveForX(*line, globalRegion->x + globalRegion->width));
        }

        // Draw the line
        kTest(GdkGraphicLineSet_Construct(&lineSet, 2.0, color, points, 2, alloc));
        kTest(GdkGraphic_AddLineSet(graphic, lineSet));
        lineSet = kNULL;
    }
    kFinally
    {
        kDestroyRef(&lineSet);
        kEndFinally();
    }

    return kOK;
}

/////////////////////////////////////////////////////////////////////////////
// Points and Vector manipulation functions
/////////////////////////////////////////////////////////////////////////////

// Convert a kPoint3d64f to a kPoint3d32f
ToolFx(kPoint3d32f) TestFeatureAverage_Point64to32(kPoint3d64f pt64)
{
    kPoint3d32f pt32;
    pt32.x = (k32f)pt64.x;
    pt32.y = (k32f)pt64.y;
    pt32.z = (k32f)pt64.z;

    return pt32;
}

// Find the average point of two points
ToolFx(kStatus) TestFeatureAverage_AveragePointPoint(kPoint3d64f pt1, kPoint3d64f pt2, kPoint3d64f* outAvgPoint)
{
    kPoint3d64f difference;

    // Find the middle point between the two points
    difference.x = pt1.x - pt2.x;
    difference.y = pt1.y - pt2.y;
    difference.z = pt1.z - pt2.z;

    outAvgPoint->x = pt2.x + difference.x / 2;
    outAvgPoint->y = pt2.y + difference.y / 2;
    outAvgPoint->z = pt2.z + difference.z / 2;

    return kOK;
}

// Find the average of a point and a line
ToolFx(kStatus) TestFeatureAverage_AveragePointLine(kPoint3d64f pt, kLine3d64f line, kPoint3d64f* outAvgPoint)
{
    kPoint3d64f perpPt = { 0, 0, 0 }, normal = { 0, 0, 0 };

    // Explanation on how to calculate the average point of a point and a line:
    // =======================================================================
    //
    //    P - point
    //    D - direction of line(unit length)
    //    A - point on the line
    //
    //    X - base of the perpendicular line
    //
    //    O - Middle of the normal (Average Point)
    //
    //          P
    //         /|
    //        / |
    //       /  O
    //      /   |
    //     /    v
    //    A-----X----->D
    //
    //    (P - A).D == | X - A |
    //
    //    X  = A + ((P - A).D)D
    //    PX = X - P
    //
    //    O  = P + PX/2


    // X = A + ((P - A).D)D
    perpPt = TestFeatureAverage_AddVectors(line.p, TestFeatureAverage_ScaleVector(TestFeatureAverage_DotProduct(TestFeatureAverage_AddVectors(pt, TestFeatureAverage_Reverse(line.p)), line.dir), line.dir));
    // PX = X - P
    normal = TestFeatureAverage_AddVectors(perpPt, TestFeatureAverage_Reverse(pt));
    // O  = P + PX/2
    *outAvgPoint = TestFeatureAverage_AddVectors(pt, TestFeatureAverage_ScaleVector(0.5, normal));

    return kOK;
}

// Find the average line of two lines
ToolFx(kStatus) TestFeatureAverage_AverageLineLine(kLine3d64f line1, kLine3d64f line2, kLine3d64f* outAvgLine)
{
    k64f m1 = 0, n1 = 0, m2 = 0, n2 = 0, t = 0;
    k64f z1 = 0, z2 = 0;

    // Check if lines are parallel
    if (TestFeatureAverage_EqualVectors(TestFeatureAverage_Normalize(line1.dir), TestFeatureAverage_Normalize(line2.dir)))
    {
        // Select a point on the first line, and find the average of it with the other line
        kCheck(TestFeatureAverage_AveragePointLine(line1.p, line2, &outAvgLine->p));
        outAvgLine->dir = line1.dir;

        // Place the average point at the center of the screen (y=0)
        if (outAvgLine->dir.y != 0)
        {
            outAvgLine->p = TestFeatureAverage_LineSolveForY(*outAvgLine, 0);
        }
        else
        {
            outAvgLine->p = TestFeatureAverage_LineSolveForX(*outAvgLine, 0);
        }
    }
    else
    {
        // Switch to cartasian
        // y = m*x + n
        m1 = line1.dir.y / line1.dir.x;
        n1 = line1.p.y - m1*line1.p.x;
        m2 = line2.dir.y / line2.dir.x;
        n2 = line2.p.y - m2*line2.p.x;

        // Find the intersect point on the XY plane
        outAvgLine->p.x = (n2 - n1) / (m1 - m2);
        outAvgLine->p.y = m1*outAvgLine->p.x + n1;

        // If the lines dont interscect in 3d, find the average Z value
        t = (outAvgLine->p.x - line1.p.x) / line1.dir.x;
        z1 = line1.p.z + t*line1.dir.z;
        t = (outAvgLine->p.x - line2.p.x) / line2.dir.x;
        z2 = line2.p.z + t*line2.dir.z;
        outAvgLine->p.z = (z1 + z2) / 2;

        // The rotation will be the average of rotations
        outAvgLine->dir = TestFeatureAverage_AddVectors(line1.dir, line2.dir);
    }

    return kOK;
}

ToolFx(kPoint3d64f) TestFeatureAverage_ScaleVector(k64f scalar, kPoint3d64f vector)
{
    kPoint3d64f outVec;
    outVec.x = scalar * vector.x;
    outVec.y = scalar * vector.y;
    outVec.z = scalar * vector.z;

    return outVec;
}

ToolFx(kPoint3d64f) TestFeatureAverage_AddVectors(kPoint3d64f vector1, kPoint3d64f vector2)
{
    kPoint3d64f outVec;
    outVec.x = vector1.x + vector2.x;
    outVec.y = vector1.y + vector2.y;
    outVec.z = vector1.z + vector2.z;

    return outVec;
}

ToolFx(kPoint3d64f) TestFeatureAverage_Reverse(kPoint3d64f vector)
{
    return TestFeatureAverage_ScaleVector(-1, vector);
}

ToolFx(kPoint3d64f) TestFeatureAverage_Normalize(kPoint3d64f vector)
{
    return TestFeatureAverage_ScaleVector(1 / sqrt(vector.x*vector.x + vector.y *vector.y + vector.z*vector.z), vector);
}

ToolFx(kBool) TestFeatureAverage_EqualVectors(kPoint3d64f vector1, kPoint3d64f vector2)
{
    return (ABS(vector1.x - vector2.x) < EPSILON &&
        ABS(vector1.y - vector2.y) < EPSILON &&
        ABS(vector1.z - vector2.z) < EPSILON);
}

ToolFx(k64f) TestFeatureAverage_DotProduct(kPoint3d64f vector1, kPoint3d64f vector2)
{
    return vector1.x*vector2.x + vector1.y*vector2.y + vector1.z*vector2.z;
}

// Find a point on the line where y is given
ToolFx(kPoint3d64f) TestFeatureAverage_LineSolveForY(kLine3d64f line, k64f y)
{
    kPoint3d64f outPt = { 0, 0, 0 };
    k64f t = 0;

    t = (y - line.p.y) / line.dir.y;
    outPt.x = line.p.x + t * line.dir.x;
    outPt.y = y;
    outPt.z = line.p.z + t * line.dir.z;

    return outPt;
}

// Find a point on the line where x is given
ToolFx(kPoint3d64f) TestFeatureAverage_LineSolveForX(kLine3d64f line, k64f x)
{
    kPoint3d64f outPt = { 0, 0, 0 };
    k64f t = 0;

    t = (x - line.p.x) / line.dir.x;
    outPt.x = x;
    outPt.y = line.p.y + t * line.dir.y;
    outPt.z = line.p.z + t * line.dir.z;

    return outPt;
}
