/////////////////////////////////////////////////////////////////////////////
// TestSurfaceDynamicAlignment
// 2017-12-12 (C) LMI Technologies
//
// Tool to test the usage of dynamic outputs
/////////////////////////////////////////////////////////////////////////////

#include <GdkAppSample/TestDynamicOutputs.h>

kBeginClassEx(Tool, TestDynamicOutputs)
    kAddVMethod(TestDynamicOutputs, kObject, VRelease)
    kAddVMethod(TestDynamicOutputs, GdkTool, VInit)
    kAddVMethod(TestDynamicOutputs, GdkTool, VName)
    kAddVMethod(TestDynamicOutputs, GdkTool, VDescribe)
    kAddVMethod(TestDynamicOutputs, GdkTool, VNewToolConfig)
    kAddVMethod(TestDynamicOutputs, GdkTool, VNewMeasurementConfig)
    kAddVMethod(TestDynamicOutputs, GdkTool, VUpdateConfig)
    kAddVMethod(TestDynamicOutputs, GdkTool, VStart)
    kAddVMethod(TestDynamicOutputs, GdkTool, VStop)
    kAddVMethod(TestDynamicOutputs, GdkTool, VProcess)
kEndClassEx()

ToolFx(const kChar*) TestDynamicOutputs_VName()
{
    return GDK_EXAMPLE_TOOL_DYNAMIC_OUTPUTS_NAME;
}

ToolFx(kStatus) TestDynamicOutputs_VDescribe(GdkToolInfo info)
{
    GdkOutputInfo mmtInfo, featureInfo, tdOutputInfo;
    k32s defaultIntDynamicCount = 2;
    k32s defaultIntNameIndex = 7;
    kBool defaultFlag = kTRUE;

    kCheck(GdkToolInfo_SetLabel(info, GDK_EXAMPLE_TOOL_DYNAMIC_OUTPUTS_LABEL));

    kCheck(GdkToolInfo_AddSourceType(info, GDK_DATA_TYPE_UNIFORM_PROFILE));
    kCheck(GdkToolInfo_AddSourceType(info, GDK_DATA_TYPE_UNIFORM_SURFACE));

    kCheck(GdkToolInfo_AddSourceOption(info, GDK_DATA_SOURCE_TOP));
    kCheck(GdkToolInfo_AddSourceOption(info, GDK_DATA_SOURCE_BOTTOM));
    kCheck(GdkToolInfo_EnableAutoVersion(info, kFALSE));

    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_MEASUREMENT, "X", "X", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_X));

    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_MEASUREMENT, "Z", "Z", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Z));

    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_MEASUREMENT, "AverageIntensity", "AverageIntensity", &mmtInfo));

    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_FEATURE_POINT, "PointFeature", "Point Feature", &featureInfo));
    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_FEATURE_LINE, "LineFeature", "Line Feature", &featureInfo));
    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_FEATURE_CIRCLE, "CircleFeature", "Circle Feature", &featureInfo));
    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_FEATURE_PLANE, "PlaneFeature", "Plane Feature", &featureInfo));

    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_UNIFORM_PROFILE, "Profile", "Profile Output", &tdOutputInfo));
    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_UNIFORM_SURFACE, "Surface", "SurfaceOutput", &tdOutputInfo));

    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_GENERIC_1, "Generic1", "Generic1 (kObject) Data", &tdOutputInfo));
    kCheck(GdkToolInfo_AddOutput(info, GDK_DATA_TYPE_GENERIC_2, "Generic2", "Generic2 (buffer) Data", &tdOutputInfo));

    kCheck(GdkToolInfo_AddParam(info, GDK_PARAM_TYPE_INT, "DynamicCount", "Number of Dynamic Outputs", &defaultIntDynamicCount, kNULL));
    kCheck(GdkToolInfo_AddParam(info, GDK_PARAM_TYPE_BOOL, "AdjMeasurements", "Adjust Dynamic Measurements", &defaultFlag, kNULL));
    kCheck(GdkToolInfo_AddParam(info, GDK_PARAM_TYPE_BOOL, "AdjFeatures", "Adjust Dynamic Features", &defaultFlag, kNULL));
    kCheck(GdkToolInfo_AddParam(info, GDK_PARAM_TYPE_BOOL, "AdjToolDataOutputs", "Adjust Dynamic Tool Data Outputs", &defaultFlag, kNULL));
    kCheck(GdkToolInfo_AddParam(info, GDK_PARAM_TYPE_INT, "NameIndex", "Index for output name retrieval", &defaultIntNameIndex, kNULL));

    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VInit(TestDynamicOutputs info, kType type, kAlloc alloc)
{
    TestDynamicOutputsClass* K_ATTRIBUTE_UNUSED obj =  (TestDynamicOutputsClass*)info;

    kCheck(GdkTool_VInit(info, type, alloc));
    obj->measurementXBaseIndex = 0;
    obj->measurementZBaseIndex = 0;
    obj->measurementAIBaseIndex = 0;

    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VRelease(TestDynamicOutputs info)
{
    TestDynamicOutputsClass* K_ATTRIBUTE_UNUSED obj =  xTestDynamicOutputs_Cast(info);

    return GdkTool_VRelease(info);
}

ToolFx(kStatus) TestDynamicOutputs_VNewToolConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VNewMeasurementConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkMeasurementCfg measurementConfig)
{
    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    GdkParam prmDynamicCount        = GdkToolCfg_FindParameter(toolConfig, "DynamicCount");
    GdkParam prmAdjustMeasurements  = GdkToolCfg_FindParameter(toolConfig, "AdjMeasurements");
    GdkParam prmAdjustFeatures      = GdkToolCfg_FindParameter(toolConfig, "AdjFeatures");
    GdkParam prmAdjustTdo           = GdkToolCfg_FindParameter(toolConfig, "AdjToolDataOutputs");
    GdkParam prmNameIndex           = GdkToolCfg_FindParameter(toolConfig, "NameIndex");

    GdkMeasurementCfg newMmtCfg = kNULL;
    GdkFeatureCfg newFeatCfg = kNULL;
    GdkToolDataOutputCfg newTdoCfg = kNULL;
    k32s i;
    kText64 outputLabel;

    kTry
    {
        k32s    dynamicCount        = GdkParam_AsInt (prmDynamicCount);
        kBool   adjustMeasurements  = GdkParam_AsBool(prmAdjustMeasurements);
        k32s    adjustFeatures      = GdkParam_AsBool(prmAdjustFeatures);
        kBool   adjustTdo           = GdkParam_AsBool(prmAdjustTdo);
        k32s    nameIndex           = GdkParam_AsInt (prmNameIndex);

        const kChar* name = "NULL";
    
        GdkToolOutputCfg outputCfg = GdkToolCfg_OutputAt(toolConfig, nameIndex);

        name = GdkToolOutputCfg_Name(outputCfg);

        kTest(GdkLogf("Output index %d name is %s", nameIndex, name));

        if (adjustMeasurements)
        {
            kText32 mmtTypeStr = "Z";

            // remove measurements with matching type except for the original one created in the VDescribe
            for (i = (k32s) GdkToolCfg_MeasurementCount(toolConfig)-1; i >= DEFINED_MEASUREMENTS_COUNT + dynamicCount; i--)
            {
                if (kStrCompare(GdkMeasurementCfg_Type(GdkToolCfg_MeasurementAt(toolConfig, i)), mmtTypeStr) == 0)
                {
                    kTest(GdkToolCfg_RemoveMeasurement(toolConfig, i));
                }
            }

            // add number of new measurements specified by the "dynamicCount" parameter
            for (i = (k32s)GdkToolCfg_MeasurementCount(toolConfig); i < DEFINED_MEASUREMENTS_COUNT + dynamicCount; i++)
            {
                kTest(GdkToolCfg_AddMeasurement(toolConfig, mmtTypeStr, &newMmtCfg));
                kStrPrintf(outputLabel, kCountOf(outputLabel), "Dynamic-%s.%d", mmtTypeStr, i - DEFINED_MEASUREMENTS_COUNT + 1);
                kTest(GdkMeasurementCfg_SetName(newMmtCfg, outputLabel));
                kTest(GdkMeasurementCfg_SetEnabled(newMmtCfg, kTRUE));
            }
        }

        if (adjustFeatures)
        {
            kText32 featTypeStr = "PointFeature";

            // remove features with matching type except for the original one created in the VDescribe
            for (i = (k32s) GdkToolCfg_FeatureCount(toolConfig)-1; i >= DEFINED_FEATURES_COUNT + dynamicCount; i--)
            {
                if (kStrCompare(GdkFeatureCfg_Type(GdkToolCfg_FeatureAt(toolConfig, i)), featTypeStr) == 0)
                {
                    kTest(GdkToolCfg_RemoveFeature(toolConfig, i));
                }
            }

            // add number of new features specified by the "dynamicCount" parameter
            for (i = (k32s)GdkToolCfg_FeatureCount(toolConfig); i < DEFINED_FEATURES_COUNT + dynamicCount; i++)
            {
                kTest(GdkToolCfg_AddFeature(toolConfig, featTypeStr, &newFeatCfg));
                kStrPrintf(outputLabel, kCountOf(outputLabel), "Dynamic-%s.%d", featTypeStr, i - DEFINED_FEATURES_COUNT + 1);
                kTest(GdkFeatureCfg_SetName(newFeatCfg, outputLabel));
                kTest(GdkFeatureCfg_SetEnabled(newFeatCfg, kTRUE));
            }
        }

        if (adjustTdo)
        {
            kText32 tdoTypeStr = "Profile";

            // remove data output with matching type except for the original one created in the VDescribe
            for (i = (k32s) GdkToolCfg_ToolDataOutputCount(toolConfig)-1; i >= DEFINED_TOOLDATAOUTPUTS_COUNT + dynamicCount; i--)
            {
                if (kStrCompare(GdkToolDataOutputCfg_Type(GdkToolCfg_ToolDataOutputAt(toolConfig, i)), tdoTypeStr) == 0)
                {
                    kTest(GdkToolCfg_RemoveToolDataOutput(toolConfig, i));
                }
            }

            // add number of new tool data outputs specified by the "dynamicCount" parameter
            for (i = (k32s)GdkToolCfg_ToolDataOutputCount(toolConfig); i < DEFINED_TOOLDATAOUTPUTS_COUNT + dynamicCount; i++)
            {
                kTest(GdkToolCfg_AddToolDataOutput(toolConfig, tdoTypeStr, &newTdoCfg));
                kStrPrintf(outputLabel, kCountOf(outputLabel), "Dynamic-%s.%d", tdoTypeStr, i - DEFINED_TOOLDATAOUTPUTS_COUNT + 1);
                kTest(GdkToolDataOutputCfg_SetName(newTdoCfg, outputLabel));
                kTest(GdkToolDataOutputCfg_SetEnabled(newTdoCfg, kTRUE));
            }
        }
    }
    kFinally
    {
        kEndFinally();
    }

    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VStart(TestDynamicOutputs tool)
{
    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VStop(TestDynamicOutputs tool)
{
    return kOK;
}

ToolFx(kStatus) TestDynamicOutputs_VProcess(TestDynamicOutputs tool, GdkToolInput input, GdkToolOutput output)
{
    TestDynamicOutputsClass* K_ATTRIBUTE_UNUSED obj =  (TestDynamicOutputsClass*)tool;
    kStatus exception = kOK;
    kSize measurementsCount = GdkToolCfg_MeasurementCount(obj->base.config);
    kSize featuresCount = GdkToolCfg_FeatureCount(obj->base.config);
    kSize tdosCount = GdkToolCfg_ToolDataOutputCount(obj->base.config);
    GvPointFeatureMsg pointFeatureMsg = kNULL;
    GvLineFeatureMsg lineFeatureMsg = kNULL;
    GvCircleFeatureMsg circleFeatureMsg = kNULL;
    GvPlaneFeatureMsg planeFeatureMsg = kNULL;
    GvMeasureMsg measurementMsg = kNULL;
    GvProfileMsg profileMsg = kNULL;
    kPoint3d64f position, normal;
    kPoint3d64f zeroPosition = { 0.5, 0.6, 0.7 };
    kPoint3d64f onePosition = { 1.0, 1.0, 1.0 };
    kSize i;

    kTry
    {
        // Measurements
        kTest(GdkToolOutput_InitMeasurementAt(output, 0, &measurementMsg));
        if (measurementMsg)
        {
            kTest(GvMeasureMsg_SetValue(measurementMsg, 1.1));
            kTest(GvMeasureMsg_SetStatus(measurementMsg, GDK_MEASUREMENT_OK));
            kTest(GvMeasureMsg_SetResultPosition(measurementMsg, &zeroPosition));
        }

        kTest(GdkToolOutput_InitMeasurementAt(output, 1, &measurementMsg));
        if (measurementMsg)
        {
            kTest(GvMeasureMsg_SetValue(measurementMsg, 2.2));
            kTest(GvMeasureMsg_SetStatus(measurementMsg, GDK_MEASUREMENT_OK));
            kTest(GvMeasureMsg_SetResultPosition(measurementMsg, &onePosition));
        }

        kTest(GdkToolOutput_InitMeasurementAt(output, 2, &measurementMsg));
        if (measurementMsg)
        {
            kTest(GvMeasureMsg_SetValue(measurementMsg, 3.3));
            kTest(GvMeasureMsg_SetStatus(measurementMsg, GDK_MEASUREMENT_OK));
            kTest(GvMeasureMsg_SetResultPosition(measurementMsg, &onePosition));
        }

        for (i = 0; i < (kSize) (measurementsCount - DEFINED_MEASUREMENTS_COUNT); i++)
        {
            kTest(GdkToolOutput_InitMeasurementAt(output, DEFINED_MEASUREMENTS_COUNT+i, &measurementMsg));
            if (measurementMsg)
            {
                kTest(GvMeasureMsg_SetValue(measurementMsg, 5.5));
                kTest(GvMeasureMsg_SetStatus(measurementMsg, GDK_MEASUREMENT_OK));
                kTest(GvMeasureMsg_SetResultPosition(measurementMsg, &onePosition));
            }
        }

        // Features
        position.x = 10; position.y = 20; position.z = 30;
        normal.x = -0.643; normal.y = 0.49191666; normal.z = -0.587;

        kTest(GdkToolOutput_InitFeatureAt(output, measurementsCount, &pointFeatureMsg));
        if (pointFeatureMsg)
        {
            kTest(GvFeatureMsg_SetPosition(pointFeatureMsg, &position));
        }

        kTest(GdkToolOutput_InitFeatureAt(output, measurementsCount+1, &lineFeatureMsg));
        if (lineFeatureMsg)
        {
            kTest(GvFeatureMsg_SetPosition(lineFeatureMsg, &position));
            kTest(GvFeatureMsg_SetOrientation(lineFeatureMsg, &normal));
        }

        kTest(GdkToolOutput_InitFeatureAt(output,  measurementsCount+2, &circleFeatureMsg));
        if (circleFeatureMsg)
        {
            kTest(GvCircleFeatureMsg_SetRadius(circleFeatureMsg, 10));
            kTest(GvCircleFeatureMsg_SetPosition(circleFeatureMsg, &position));
            kTest(GvCircleFeatureMsg_SetNormal(circleFeatureMsg, &normal));
        }

        kTest(GdkToolOutput_InitFeatureAt(output,  measurementsCount+3, &planeFeatureMsg));
        if (planeFeatureMsg)
        {
            kTest(GvPlaneFeatureMsg_SetNormal(planeFeatureMsg, &normal));
            kTest(GvPlaneFeatureMsg_SetOriginDistance(planeFeatureMsg, 10));
        }

        for (i = 0; i < (kSize) (featuresCount-DEFINED_FEATURES_COUNT); i++)
        {
            kTest(GdkToolOutput_InitFeatureAt(output,  measurementsCount+DEFINED_FEATURES_COUNT+i, &pointFeatureMsg));
            if (pointFeatureMsg)
            {
                position.x = (k64f) (5 * i);
                position.z = (k64f) (5 * i);
                kTest(GvFeatureMsg_SetPosition(pointFeatureMsg, &position));
            }
        }

        // Tool data outputs
        kTest(GdkToolOutput_InitProfileAt(output, measurementsCount+featuresCount, 100, &profileMsg));
        if (profileMsg)
        {
            k16s* dataItem;

            dataItem = (k16s*)GvProfileMsg_Points(profileMsg);

            for (i = 0; i < 100; i++)
            {
                dataItem[i] = 50;
            }
        }

        // skip populating the surface and 2 generic outputs

        // dynamic tool data outputs
        for (i = 0; i < (kSize) (tdosCount - DEFINED_TOOLDATAOUTPUTS_COUNT); i++)
        {
            kTest(GdkToolOutput_InitProfileAt(output, measurementsCount+featuresCount+DEFINED_TOOLDATAOUTPUTS_COUNT+i, 1, &profileMsg));
            if (profileMsg)
            {
                k16s* dataItem;

                dataItem = (k16s*)GvProfileMsg_Points(profileMsg);
                dataItem[0] = 7;
            }
        }
    }
    kCatch(&exception)
    {
        kEndCatch(exception);
    }

    return kOK;
}
