/**
 * @file    GdkDef.h
 * @brief   Essential Gdk declarations.
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef GDK_DEF_H
#define GDK_DEF_H

#include <kApi/kApiDef.h>
#include <kFireSync/kFsDef.h>

#if defined(GDK_EMIT)
#    define GdkFx(TYPE)       kExportFx(TYPE)
#    define GdkDx(TYPE)       kExportDx(TYPE)
#    define GdkCppFx(TYPE)    kExportDx(TYPE)
#    define GdkClass          kExportClass
#else
#    define GdkFx(TYPE)       kImportFx(TYPE)
#    define GdkDx(TYPE)       kImportDx(TYPE)
#    define GdkCppFx(TYPE)    kImportDx(TYPE)
#    define GdkClass          kImportClass
#endif

kDeclareValueEx(Gdk, GdkStreamId, kValue)

#define GDK_MAX_SENSOR_COUNT 32  // Should correspond to GS_MAX_SENSOR_COUNT

typedef struct GdkStreamId 
{
    k32s step;
    k32s id;
    k32s source;
} GdkStreamId;

typedef struct GdkRegion2d64f
{
    k64f x;
    k64f y;
    k64f width;
    k64f length;
} GdkRegion2d64f;

typedef struct GdkRegion3d64f
{
    k64f x;
    k64f y;
    k64f z;
    k64f width;
    k64f length;
    k64f height;
    k64f zAngle;
} GdkRegion3d64f;

typedef struct GdkRegionXZ64f
{
    k64f x;
    k64f z;
    k64f width;
    k64f height;
} GdkRegionXZ64f;

typedef struct GdkPointXZ16s
{
    k16s x;
    k16s z;
} GdkPointXZ16s;

/**
* @struct  GdkPointSetRegionMode
* @extends kValue
* @ingroup Gdk
* @brief   Connectedness of points in point set region.
*/
typedef k32s GdkPointSetRegionMode;

/** @relates GdkPointSetRegionMode @{ */
#define GDK_POINT_SET_MODE_POINTS           (1)             ///< Individual points
#define GDK_POINT_SET_MODE_OPEN             (2)             ///< Open, connected in order
#define GDK_POINT_SET_MODE_CLOSED           (3)             ///< Closed loop
/** @} */

/**
* @struct  GdkPointSetRegionColor
* @extends kValue
* @ingroup Gdk
* @brief   Color, of points or connecting lines.
*/
typedef k32u GdkPointSetRegionColor;

/** @relates GdkPointSetRegionColor @{ */
#define GDK_POINT_SET_COLOR_BLACK               (0x000000)             ///< Black
#define GDK_POINT_SET_COLOR_BLUE                (0x0000ff)             ///< Blue
#define GDK_POINT_SET_COLOR_CYAN                (0x00ffff)             ///< Cyan
#define GDK_POINT_SET_COLOR_DARK_RED            (0x720000)             ///< Dark red
#define GDK_POINT_SET_COLOR_GREEN               (0x00ff00)             ///< Green
#define GDK_POINT_SET_COLOR_LIGHT_BLUE          (0x4169ff)             ///< Light blue
#define GDK_POINT_SET_COLOR_MAGENTA             (0xff00ff)             ///< Magenta
#define GDK_POINT_SET_COLOR_ORANGE              (0xffbb00)             ///< Orange
#define GDK_POINT_SET_COLOR_PINK                (0xffbdc1)             ///< Pink
#define GDK_POINT_SET_COLOR_RED                 (0xff0000)             ///< Red
#define GDK_POINT_SET_COLOR_WHITE               (0xffffff)             ///< White
#define GDK_POINT_SET_COLOR_YELLOW              (0xffff00)             ///< Yellow
#define GDK_POINT_SET_COLOR_PURPLE              (0x7f007f)             ///< Purple
#define GDK_POINT_SET_COLOR_GRAY                (0x3f3f3f)             ///< Gray
#define GDK_POINT_SET_COLOR_LIGHT_GRAY          (0xd3d3d3)             ///< Light gray
#define GDK_POINT_SET_COLOR_YELLOW50_BLACK50    (0x808000)             ///< 50/50 yellow/black
#define GDK_POINT_SET_COLOR_LIGHT_GREEN         (0x67D0C5)             ///< Light green
#define GDK_POINT_SET_COLOR_DARK_ORANGE         (0xD26500)             ///< Dark orange
#define GDK_POINT_SET_COLOR_VIOLET              (0xCCB5F1)             ///< Violet
#define GDK_POINT_SET_COLOR_BRIGHT_PURPLE       (0x9064FF)             ///< Bright purple
#define GDK_POINT_SET_COLOR_BRIGHT_RED          (0xA10000)             ///< Bright red
#define GDK_POINT_SET_COLOR_MUSTARD             (0x868C02)             ///< Mustard
#define GDK_POINT_SET_COLOR_DARK_MUSTARD        (0x484900)             ///< Dark mustard
/** @} */

/**
* @struct  GdkTransform3d
* @extends kValue
* @ingroup Gdk
* @brief   3D transformation matrix

[ xx xy xz xt ]
[ yx yy yz yt ]
[ zx zy zz zt ]
[ 0  0  0  1  ]
*/

/** @relates GdkTransform3d @{ */
typedef struct GdkTransform3d 
{
    k64f xx;
    k64f xy;
    k64f xz;
    k64f xt;

    k64f yx;
    k64f yy;
    k64f yz;
    k64f yt;

    k64f zx;
    k64f zy;
    k64f zz;
    k64f zt;

} GdkTransform3d;
/** @} */


GdkFx(kStatus) GdkRegion3d64f_FromRect(GdkRegion3d64f* region, const kRect3d64f* rect);
GdkFx(kStatus) GdkRegion3d64f_ToRect(const GdkRegion3d64f* region, kRect3d64f* rect);

/**
* @struct  GdkOrientation
* @extends kValue
* @ingroup Gdk
* @brief   Represents the sensor system orientation.
*/
typedef k32s GdkOrientation;

/** @relates GdkOrientation @{ */
#define GDK_ORIENTATION_WIDE         (0)             ///< Wide layout.
#define GDK_ORIENTATION_OPPOSITE     (1)             ///< Opposite layout.
#define GDK_ORIENTATION_REVERSE      (2)             ///< Reverse layout.
/** @} */

/**
* @struct  GdkDataSource
* @extends kValue
* @ingroup Gdk
* @brief   Represents a data source selection.
*/
typedef k32s GdkDataSource;

/** @relates GdkDataSource @{ */
#define GDK_DATA_SOURCE_NONE                        (-1)            ///< None or invalid.
#define GDK_DATA_SOURCE_TOP                         (0)             ///< Top.
#define GDK_DATA_SOURCE_BOTTOM                      (1)             ///< Bottom.
#define GDK_DATA_SOURCE_TOP_LEFT                    (2)             ///< Top left.
#define GDK_DATA_SOURCE_TOP_RIGHT                   (3)             ///< Top right.
#define GDK_DATA_SOURCE_TOP_BOTTOM                  (4)             ///< Top and bottom (both data streams).
#define GDK_DATA_SOURCE_LEFT_RIGHT                  (5)             ///< Left and right (both data streams).
#define GDK_DATA_SOURCE_COUNT                       (6)             ///< Reserved for internal use.
#define GDK_DATA_SOURCE_TOOLDATAOUTPUT              (GDK_DATA_SOURCE_TOP)    ///< Treat Tool Data Output source as Top.
#define GDK_DATA_SOURCE_DEVICE_BASE                 (100)           ///< Base value to retrieve a buddied device's data stream. Source id is base + device index (where index is from 0..GDK_MAX_SENSOR_COUNT-1, and index 0 is the main sensor).
#define GDK_DATA_SOURCE_DEVICE_LAST                 (GDK_DATA_SOURCE_DEVICE_BASE + GDK_MAX_SENSOR_COUNT - 1) ///< Last buddy device in a buddy configuration.
#define GDK_DATA_SOURCE_DEVICE_ALL                  (GDK_DATA_SOURCE_DEVICE_LAST + 1)                        ///< Allow all sensors in a buddy configuration as a data stream source.
#define GDK_DATA_SOURCE_IS_DEVICE_SOURCE(sourceId)  (((sourceId) >= GDK_DATA_SOURCE_DEVICE_BASE) && ((sourceId) <= GDK_DATA_SOURCE_DEVICE_LAST)) ///< Macro to check if a source is a device in a buddy configuration.

/** @} */

/**
 * @ingroup Gdk
 * @defgroup RESERVED Reserved
 * @brief   Reserved Gdk declarations.
 * @{
 */
#define GDK_DATA_PRIMARY_INPUT_NAME             "PrimaryData"   ///< Index 0 is reserved for the Primary Data Input name. 
#define GDK_DATA_PRIMARY_INPUT_INDEX            0               ///< Primary input message index
#define GDK_DATA_SECONDARY_INPUT_START_INDEX    (GDK_DATA_PRIMARY_INPUT_INDEX + 1)  ///< Secondary input (eg. parameters, features) index starting value.
/** @} */

/**
* @struct  GdkDataType
* @extends kValue
* @ingroup Gdk
* @brief   Represents data type. No longer used as a bitmask.
*          Note that generic data types will have their msb set to 1 
*          (ie. start from 0x80000000).
*          The first 17 generic data types are defined below for convenience.
*/
typedef k32s GdkDataType;

/** @relates GdkDataType @{ */
#define GDK_DATA_TYPE_NONE                          (0x1)         ///< None.
#define GDK_DATA_TYPE_RANGE                         (0x2)         ///< Range data.
#define GDK_DATA_TYPE_UNIFORM_PROFILE               (0x3)         ///< Uniformly-spaced (resampled) profile data.
#define GDK_DATA_TYPE_PROFILE_POINT_CLOUD           (0x4)         ///< Profile point cloud.
#define GDK_DATA_TYPE_UNIFORM_SURFACE               (0x5)         ///< Uniformly-spaced height map. (Points = k16s)
#define GDK_DATA_TYPE_SURFACE_POINT_CLOUD           (0x6)         ///< Surface point cloud. (Points = kPoint3d16s)
#define GDK_DATA_TYPE_UNMERGED_PROFILE_POINT_CLOUD  (0x7)         ///< Unmerged profile point cloud data. Reserved for future use.
#define GDK_DATA_TYPE_VIDEO                         (0x8)         ///< Video data.
#define GDK_DATA_TYPE_TRACHEID                      (0x9)         ///< Tracheid data. Reserved for internal use.
#define GDK_DATA_TYPE_MEASUREMENT                   (0xA)         ///< Measurement data
#define GDK_DATA_TYPE_MESH                          (0xB)         ///< Mesh data

#define GDK_DATA_TYPE_FEATURES                      (0x200)       ///< Geometric features only. No scan data.
#define GDK_DATA_TYPE_FEATURE_POINT                 (0x201)       ///< Point feature data
#define GDK_DATA_TYPE_FEATURE_LINE                  (0x202)       ///< Line feature data
#define GDK_DATA_TYPE_FEATURE_CIRCLE                (0x203)       ///< Circle feature data
#define GDK_DATA_TYPE_FEATURE_PLANE                 (0x204)       ///< Plane feature data

#define GDK_DATA_TYPE_GENERIC_BASE                  (0x80000000)                      ///< Generic data base
#define GDK_DATA_TYPE_GENERIC_1                     (GDK_DATA_TYPE_GENERIC_BASE+1)    ///< Generic data 1
#define GDK_DATA_TYPE_GENERIC_2                     (GDK_DATA_TYPE_GENERIC_BASE+2)    ///< Generic data 2
#define GDK_DATA_TYPE_GENERIC_3                     (GDK_DATA_TYPE_GENERIC_BASE+3)    ///< Generic data 3
#define GDK_DATA_TYPE_GENERIC_4                     (GDK_DATA_TYPE_GENERIC_BASE+4)    ///< Generic data 4
#define GDK_DATA_TYPE_GENERIC_5                     (GDK_DATA_TYPE_GENERIC_BASE+5)    ///< Generic data 5
#define GDK_DATA_TYPE_GENERIC_6                     (GDK_DATA_TYPE_GENERIC_BASE+6)    ///< Generic data 6
#define GDK_DATA_TYPE_GENERIC_7                     (GDK_DATA_TYPE_GENERIC_BASE+7)    ///< Generic data 7
#define GDK_DATA_TYPE_GENERIC_8                     (GDK_DATA_TYPE_GENERIC_BASE+8)    ///< Generic data 8
#define GDK_DATA_TYPE_GENERIC_9                     (GDK_DATA_TYPE_GENERIC_BASE+9)    ///< Generic data 9
#define GDK_DATA_TYPE_GENERIC_10                    (GDK_DATA_TYPE_GENERIC_BASE+10)   ///< Generic data 10
#define GDK_DATA_TYPE_GENERIC_11                    (GDK_DATA_TYPE_GENERIC_BASE+11)   ///< Generic data 11
#define GDK_DATA_TYPE_GENERIC_12                    (GDK_DATA_TYPE_GENERIC_BASE+12)   ///< Generic data 12
#define GDK_DATA_TYPE_GENERIC_13                    (GDK_DATA_TYPE_GENERIC_BASE+13)   ///< Generic data 13
#define GDK_DATA_TYPE_GENERIC_14                    (GDK_DATA_TYPE_GENERIC_BASE+14)   ///< Generic data 14
#define GDK_DATA_TYPE_GENERIC_15                    (GDK_DATA_TYPE_GENERIC_BASE+15)   ///< Generic data 15
#define GDK_DATA_TYPE_GENERIC_16                    (GDK_DATA_TYPE_GENERIC_BASE+16)   ///< Generic data 16
#define GDK_DATA_TYPE_GENERIC_END                   (0xFFFFFFFF)                      ///< Generic data last available

#define IS_GDK_DATA_TYPE_GENERIC(dataType) \
            ((GdkDataType) dataType >= (GdkDataType) GDK_DATA_TYPE_GENERIC_BASE && (GdkDataType) dataType <= (GdkDataType) GDK_DATA_TYPE_GENERIC_END)

#define GDK_DATA_TYPE_FEATURES_ONLY        (GDK_DATA_TYPE_FEATURES)                     ///< Geometric features only. (DEPRECATED)
#define GDK_DATA_TYPE_RAW_PROFILE          (GDK_DATA_TYPE_PROFILE_POINT_CLOUD)          ///< Unresampled profile data. (DEPRECATED)
#define GDK_DATA_TYPE_RAW_SURFACE          (GDK_DATA_TYPE_SURFACE_POINT_CLOUD)          ///< Unresampled point cloud. (Points = kPoint3d16s) (DEPRECATED)
#define GDK_DATA_TYPE_UNMERGED_RAW_PROFILE (GDK_DATA_TYPE_UNMERGED_PROFILE_POINT_CLOUD) ///< Unmerged raw profile data. (DEPRECATED)
/** @} */

/**
* @struct  GdkGraphicAnchorRegionType
* @extends kValue
* @ingroup Gdk
* @brief   Represents type of anchor region.
*/
typedef k32s GdkGraphicAnchorRegionType;

/** @relates GdkGraphicAnchorRegionType @{ */
#define GDK_GRAPHIC_ANCHOR_REGION_2D        (0x0)
#define GDK_GRAPHIC_ANCHOR_REGION_3D        (0x1)
#define GDK_GRAPHIC_ANCHOR_REGION_2D_XY     (0x2)
/** @} */

/**
* @struct  GdkGraphicPositionType
* @extends kValue
* @ingroup Gdk
* @brief   Represents type of GDK Graphic Position.
*/
typedef k32s GdkGraphicPositionType;

/** @relates GdkGraphicPositionType @{ */
#define GDK_GRAPHIC_POSITION_TYPE_NONE      (0x0)   ///< No Graphic position.
#define GDK_GRAPHIC_POSITION_TYPE_X         (0x1)   ///< Graphic position X.
#define GDK_GRAPHIC_POSITION_TYPE_Y         (0x2)   ///< Graphic position Y.
#define GDK_GRAPHIC_POSITION_TYPE_Z         (0x3)   ///< Graphic position Z.
/** @} */

/**
* @struct  GdkAnchorParam
* @extends kValue
* @ingroup Gdk
* @brief   Represents anchor parameter source selections.
*/
typedef k32s GdkAnchorParam;

/** @relates GdkAnchorParam @{ */
#define GDK_ANCHOR_PARAM_X          (0)         ///< X Axis.
#define GDK_ANCHOR_PARAM_Y          (1)         ///< Y Axis.
#define GDK_ANCHOR_PARAM_Z          (2)         ///< Z Axis.
#define GDK_ANCHOR_PARAM_Z_ANGLE    (3)         ///< Z Rotation Angle.
#define GDK_ANCHOR_PARAM_COUNT      (4)         ///< Reserved for internal use.
/** @} */

GdkFx(const kChar*) GdkAnchorParam_Format(GdkAnchorParam axis);

/**
 * @struct GdkAxis
 * @extends kValue
 * @ingroup Gdk
 * @brief   Represents axis X, Y, Z.
 */
typedef k32s GdkAxis;
/** @relates GdkAxis @{ */
#define GDK_AXIS_X                   (0)        ///< X Axis.
#define GDK_AXIS_Y                   (1)        ///< Y Axis.
#define GDK_AXIS_Z                   (2)        ///< Z Axis.
/** @} */

/**
* @struct  GdkRefFrame
* @extends kValue
* @ingroup Gdk
* @brief   Represents data source selections.
*/
typedef k32s GdkRefFrame;

/** @relates GdkRefFrame @{ */
#define GDK_REF_FRAME_SENSOR         (0)        ///< Sensor frame of reference.
#define GDK_REF_FRAME_SCAN           (1)        ///< Scan frame of reference (X in sensor frame, Y in part frame).
#define GDK_REF_FRAME_PART           (2)        ///< Part frame of reference.
/** @} */

/**
* @struct  GdkMeasurementValueType
* @extends kValue
* @ingroup Gdk
* @brief   Represents data source selections.
*/
typedef k32s GdkMeasurementValueType;

/** @relates GdkMeasurementValueType @{ */
#define GDK_MEASUREMENT_VALUE_TYPE_VALUE     (0)         ///< Generic value.
#define GDK_MEASUREMENT_VALUE_TYPE_X         (1)         ///< X value (mm).
#define GDK_MEASUREMENT_VALUE_TYPE_Y         (2)         ///< Y value (mm).
#define GDK_MEASUREMENT_VALUE_TYPE_Z         (3)         ///< Z value (mm).
#define GDK_MEASUREMENT_VALUE_TYPE_X_ANGLE   (4)         ///< X angle (degrees).
#define GDK_MEASUREMENT_VALUE_TYPE_Y_ANGLE   (5)         ///< Y angle (degrees).
#define GDK_MEASUREMENT_VALUE_TYPE_Z_ANGLE   (6)         ///< Z angle (degrees).
/** @} */

/**
* @struct  GdkToolOutputType
* @extends k32s
* @ingroup Gdk
* @brief   Represents tool output type.
*/
typedef k32s GdkToolOutputType;

/** @relates GdkToolOutputType @{ */
#define GDK_TOOL_OUTPUT_TYPE_NULL            (0)     ///< Null output.
#define GDK_TOOL_OUTPUT_TYPE_MEASUREMENT     (1)     ///< Measurement output.
#define GDK_TOOL_OUTPUT_TYPE_FEATURE         (2)     ///< Feature output.
#define GDK_TOOL_OUTPUT_TYPE_DIAGNOSTIC      (3)     ///< Diagnostic output.
#define GDK_TOOL_OUTPUT_TYPE_TOOLDATA_OUTPUT (4)     ///< Tool Data output.
/** @} */


/**
* @struct  GdkFeatureDataType
* @extends k32s
* @ingroup Gdk
* @brief   Represents geometric feature output type.
*/
typedef k32s GdkFeatureDataType;
/** @relates GdkFeatureDataType @{ */
#define GDK_FEATURE_TYPE_NULL   (-1)                ///< Null feature.
#define GDK_FEATURE_TYPE_POINT  (0)                 ///< Point feature.
#define GDK_FEATURE_TYPE_LINE   (1)                 ///< Line feature.
#define GDK_FEATURE_TYPE_CIRCLE (2)                 ///< Circle feature.
#define GDK_FEATURE_TYPE_PLANE  (3)                 ///< Plane feature.
/** @} */
kDeclareEnumEx(Gdk, GdkFeatureDataType, kValue)

typedef k32s GdkGeometricFeatureId;

#define GDK_GEOMETRIC_FEATURE_ID_NONE       (-1)

GdkFx(const kChar*) GdkFeatureDataType_ToString(GdkFeatureDataType type);
GdkFx(GdkFeatureDataType) GdkFeatureDataType_ToType(const kChar* typeName);

typedef k32s GdkToolFormat;
#define GDK_TOOL_FORMAT_STANDARD (0)
#define GDK_TOOL_FORMAT_USER     (1)
#define GDK_TOOL_FORMAT_INTERNAL (2)

typedef struct GdkMeasurementData 
{
    kPoint3d64f anchorPosition;
} GdkMeasurementData;

GdkFx(kStatus) GdkLogf(const kChar* message, ...);

// Forward declarations

typedef kObject GdkTool;

#endif
